module precision 
 
integer, parameter :: kr4 = selected_real_kind(6,37)      
integer, parameter :: kr8 = selected_real_kind(15,307)  
integer, parameter :: kr16 = selected_real_kind(30,1000) 
integer, parameter :: ki4 = selected_int_kind(9)           
integer, parameter :: ki8 = selected_int_kind(18)          
integer, parameter :: kc4 = kr4                            
integer, parameter :: kc8 = kr8                            
! generic kinds
integer, parameter :: ki=ki4,kr=kr8,kc=kc8
 
end module precision

module pdb
  use precision
  implicit none
  
  integer :: n ! number of atoms in pdb
  type :: label ! define new type label
     character*6 :: recrd
     integer     :: serial
     character*4 :: name
     character*1 :: altloc
     character*3 :: resname
     character*1 :: chainid
     integer     :: resseq
     character*1 :: icode
  end type label
  type(label), allocatable :: pdb_label(:)
  real(kr), allocatable   :: x(:), y(:), z(:)
  
contains
  
  subroutine alloc_pdb
    ! allocate memory for data from a n-atoms pdb 
    use precision
    implicit none
    integer :: err
    
    allocate(x(n),y(n),z(n),pdb_label(n),stat=err)
    if(err /= 0) then 
       write(0,*) 'problem in allocating memory for pdb data'
       stop
    endif
    
  end subroutine alloc_pdb
  
  subroutine dealloc_pdb
    ! deallocate memory for data from a n-atoms pdb 
    use precision
    implicit none
    integer :: err
    
    deallocate(x,y,z,pdb_label,stat=err)
    if(err /= 0) then 
       write(0,*) 'problem in deallocating memory for pdb data'
       stop
    endif
    
  end subroutine dealloc_pdb
    
  
  subroutine read_pdb(pdbin)
    ! read a pdb file from unit pdbin
    use precision
    implicit none
    integer(ki), intent(in) :: pdbin ! unit number
    ! local variables
    integer(ki) :: i,k
    integer(ki) :: ierror
    integer(ki) :: ires1
    character   :: at1*4,res1*3
    character   :: line*64,atom*6
    real(kr)    :: x1,y1,z1

    ! count the number of atoms
    
    n = 0
    atom_count: do 
       read(pdbin,*,iostat=ierror) line
       if(ierror < 0 .or. line(1:3) == 'END') exit atom_count
       if(line(1:3) == 'ATO') n = n + 1
    enddo atom_count
    write(0,*) 'reference pdb contains ',n,' atoms'
    
    ! check for n greater than zero
    
    if(n == 0) then 
       write(0,*) 'no ATOM keyword in the pdb file?'
       stop
    endif
    
    ! allocate memory for pdb data
    
    call alloc_pdb
    
    ! rewind pdb file 
    
    rewind(pdbin)
    
    ! skip the first lines of comment
    
    skip_comment: do 
       read(pdbin,*,iostat=ierror) line
       if(line(1:3) == 'ATO') then 
          backspace(pdbin)
          exit skip_comment
       endif
    end do skip_comment
    
    ! read pdb data
    
    atom_read: do i = 1,n
       
       read(pdbin,1,iostat=ierror) pdb_label(i),x(i),y(i),z(i)
       if(ierror > 0) then 
          write(0,*) 'fatal error at line : ', i
          write(0,*) 'check that all fields of pdb file are separated'
          stop
       elseif(ierror < 0) then 
          write(0,*) 'fatal error: unexpected end of pdb'
          stop
       endif
       
    enddo atom_read
    
1   format(a6,i5,1x,a4,a1,a3,1x,a1,i4,a1,3x,3f8.3,2f6.2)

    
  end subroutine read_pdb
  
  subroutine print_pdb(pdbout)
    ! read a pdb file from unit pdbin
    use precision
    implicit none
    integer(ki), intent(in) :: pdbout ! unit number
    ! local 
    integer(ki) :: i
    
    
    do i = 1,n
       write(pdbout,1) pdb_label(i), x(i), y(i), z(i)
    enddo
    write(pdbout,'(a3)') 'END'
    
1   format(a6,i5,1x,a4,a1,a3,1x,a1,i4,a1,3x,3f8.3,2f6.2)
    
  end subroutine print_pdb
  
end module pdb

program xyz2pdb
  use precision
  use pdb
  implicit none
  integer(ki)            :: i,j,err
  integer(ki)            :: narg,iargc
  integer(ki)            :: npdb
  integer(ki), parameter :: ixyz=20,ipdb=21
  integer(ki)            :: index,nat,print_index
  real(kr)               :: fstep
  real(kr)               :: co(3,3)
  real(kr)               :: factor(3),pot(3)
  real(kr)               :: x1,y1,z1
  character*128           :: arg,xyzfile,pdbfile,remarg
  logical                :: rem
  character*7            :: dummy_char

  ! find the number of arguments passed on the command line
  narg = iargc()
  
  ! initialize rem variable 
  rem = .FALSE.
  print_index = 999
  
  select case(narg)
  case(2)
     call getarg(1,xyzfile) ! .xyz file
     call getarg(2,pdbfile) ! .pdb reference file 
  case(3)
     call getarg(3,arg)
     read(arg,*) remarg
     if (remarg(1:3) == 'REM') then ! read a xyz file from a REM simulation
        rem = .TRUE. 
     else
        read(arg,*,iostat=err) print_index
        if(err /= 0) then 
           write(0,*) 'syntax: xyz2pdb <xyz_file> <reference_pdb_file> [ <replica-index> | "REM" ]'
           stop  
        else
           rem = .TRUE. 
        end if
     end if
     call getarg(1,xyzfile) ! .xyz file
     call getarg(2,pdbfile) ! .pdb reference file 
  case default
           write(0,*) 'syntax: xyz2pdb <xyz_file> <reference_pdb_file> [ <replica-index> | "REM" ]'
     stop     
  end select
  
  ! open units
  write(0,*) '.xyz file: ', xyzfile
  write(0,*) 'reference .pdb file: ', pdbfile
  
  open(unit=ixyz,file=xyzfile)
  open(unit=ipdb,file=pdbfile)
  
  ! read reference pdb file 
  call read_pdb(ipdb)
  
  npdb = 0
  
  ! read .xyz file
  read_loop: do 
     
     ! the initial lines of a .xyz file must contain
     ! -----1) number of atoms
     ! -----2) time 
     ! -----3) co matrix
     ! if the file was produced from a REM simulation, 
     ! it will contain ONE more line
     ! -----4) replica index
     read (ixyz,*,iostat=err) nat
     if (err < 0) exit read_loop
     if (nat /= n) then ! check that the number of atoms is the same in the .pdb file
        write(0,*) 'the number of atoms is ', nat
        write(0,*) 'while expecting ',n
        stop
     endif
     read (ixyz,*) fstep ! time (fs)
     read (ixyz,*) co(1,1),co(1,2),co(1,3), & ! co matrix
          co(2,1),co(2,2),co(2,3), &
          co(3,1),co(3,2),co(3,3)
     if (rem) then ! REM variables
        read (ixyz,*) dummy_char, index
     endif
     
     ! print out 
     if(index == print_index .OR. print_index == 999) then 
        write (*,3) 'REMARK number of atoms:  ', n
        write (*,4) 'REMARK simulation time:  ', fstep
        write (*,6) 'REMARK co matrix      :  ', co(1,1),co(1,2),co(1,3), &
             co(2,1),co(2,2),co(2,3), &
             co(3,1),co(3,2),co(3,3)
        if (rem) then 
           write (*,3) 'REMARK replica index  :  ', index 
           write (*,5) 'REMARK REM factors    :  ', factor
           write (*,5) 'REMARK REM energies   :  ', pot
        endif
     end if
     
     ! read the cartesian coordinates and print out in pdb format
     do i=1,n
        read (ixyz,*) x1,y1,z1
        if(index == print_index .OR. print_index == 999) write(*,1) pdb_label(i),x1,y1,z1
     end do
     if(index == print_index .OR. print_index == 999) write(*,'(a3)') 'END'
     
     npdb = npdb + 1
  enddo read_loop
  
  write(0,2) xyzfile, npdb
  
1   format(a6,i5,1x,a4,a1,a3,1x,a1,i4,a1,3x,3f8.3,2f6.2)
2 format(1x,'file ',a24,' contains ',i5,' structures')
3 format(a25,i5)
4 format(a25,f12.3)
5 format(a25,3f12.3)
6 format(a25,9f12.3)
  
end program xyz2pdb

