!--------------------------------------------------------------------------------------- 
! program to compute the ligand-receptor com-com distance in the exp pdb file 
! ligand-target COM vector are printed to standard output for each configuration. 
!--------------------------------------------------------------------------------------- 
program volume
!---------------------------------------------------------------------------------------
! The program syntax is :
! $ volume nrep filename [T/F] g
!      nrep=replica index 
!      filename+pdb file (stripped from water: e.g. 'grep -v TI file.pdb'))
!      last arg is optional and print two xyz file with full coordinate in 
!      the protein inertia frame.  
!---------------------------------------------------------------------------------------
  implicit none

  character*22 dum,filename
  character*6, allocatable :: atom(:)
  character*4, allocatable :: resd(:)
  character*6 batomi,batomj
  character*5  residue
  character*2 ires_lig
  character*6 dum6
  character*1 rep,irep,debug
  real*8   co(3,3),iner(3),mass,time,t(3,3),trot(3,3),coml(3),abserr,mltot,dd(3),dd1,new(3),vol
  real*8, allocatable :: rl(:,:),ml(:)
  real*8   rmax(3),rmin(3),boxl,dist,aux,det
  integer, allocatable ::   iresd(:)
  integer i,j,k,nrep,mrep,iret,ierr,n,nl,ires,ires_old,kmax,ilig,idum,lx,ly,lz,pme(3)
  

  abserr=0.001
  ires_old=0
  i=0
  j=0
  open(unit=10,file="tmp.pdb") 
  open(unit=11,file="setup.tmp") 
  open(unit=12,file="solvent.tmp") 
  open(unit=13,file="potential.tmp") 
  open(unit=14,file="startu.pdb") 
  open(unit=15,file="rem.tmp") 
  open(unit=16,file="rem_setup.tmp") 
  open(unit=17,file="rem_solvent.tmp") 
  open(unit=18,file="prop.tmp") 
! no REMARK should be in the file " grep -v REMARK $1 before processing with orient
  !   find natomos and nligand from first frame
  do 
     read(10,'(A22,i4)',end=10) dum,ires
!     write(6,*) dum,ires
     if(ires.eq.0) i=i+1
     j=j+1
  end do
10 nl=i
  
! now allocate coordinate and mass arrays

  allocate(rl(3,nl),ml(nl),stat=iret) 
  allocate(atom(nl),resd(nl),iresd(nl),stat=iret) 

! loop on PDB file (N.B. this must be stripped from water with e.g.: 'grep -v TIP file.pdb')

  rewind(10)
   ! read ligand coordinates
100  format(A6,i5,1x,A4,1x,A4,1x,I4,4x,3f8.3) ! PDB format
  do j=1,nl
     read(10,100) dum6,idum,atom(j),resd(j),iresd(j),(rl(k,j),k=1,3)
     call get_mass(atom(j),mass)
     ml(j)=mass
  end do
  !shift origin on com of ligand
  coml=0.d0
  mltot=0.d0
  do j=1,nl
     do k=1,3 
        coml(k)=coml(k)+ml(j)*rl(k,j)
     end do
     mltot=mltot+ml(j)
  end do
  coml=coml/mltot
  do j=1,nl
     do k=1,3
        rl(k,j)=rl(k,j)-coml(k)
     end do
  end do

! write(6,'(A4,1x,i5,3f12.5,f10.4)') 'Coml',nl,coml,mltot
  !       compute inertia tensor of the target protein
  call inertia(ml,rl,nl,t)
  t=t/1000.d0   ! divide by a 1000 to avoid jacobi hang
!  write(6,'(3f15.4)') ((t(i,j),i=1,3),j=1,3)
  call jacobi(t,trot,abserr,3)

  det =       trot(1,1)*(trot(2,2)*trot(3,3)-trot(2,3)*trot(3,2)) 
  det = det - trot(1,2)*(trot(2,1)*trot(3,3)-trot(2,3)*trot(3,1)) 
  det = det + trot(1,3)*(trot(2,1)*trot(3,2)-trot(2,2)*trot(3,1))

  if(det.lt.0) then
     do i=1,3
        trot(i,3)=-trot(i,3)
     end do
  endif
  
  coml=0.d0
  !      compute com vector of ligand
  mltot=0.d0
  rmax = -1000.d0
  rmin =  1000.d0
!  write(6,'(3f8.3)') ((trot(i,j),i=1,3),j=1,3), det 
  do j=1,nl
     call rotate(rl(1,j),trot)  ! report lig coordinate in the TARGET inertia frame 
     write(14,100) "HETATM",j,atom(j),resd(j),iresd(j),(rl(k,j),k=1,3)
     do k=1,3
        if(rl(k,j).gt.rmax(k)) rmax(k)=rl(k,j)
        if(rl(k,j).lt.rmin(k)) rmin(k)=rl(k,j)
     end do
  end do

  boxl=-1000.d0
  do i=1,3
     aux = rmax(i)-rmin(i) + 17.d0
     if(aux.gt.boxl) boxl=aux
  end do

  if(boxl.lt.30.d0) boxl=30.d0
!  write(6,*) "r-rmax", boxl

!--------------------SETUP NAMELIST --------------------------------------
  write(11,'(A)') "#&T NTHREADS    8   CACHELINE   16 "
  write(11,'(A)') "#&T NT-LEVEL1   2   CACHELINE   16 "
  write(11,'(A)') "#&T NT-LEVEL2   4   CACHELINE   16 "
  write(11,'(A)') "############################################################### "
  write(11,'(A)') "#  Minimize Crystallographic structure form PDBank  "
  write(11,'(A)') "############################################################### "
  write(11,'(A)') "  "
  write(11,'(A)') "# "
  write(11,'(A)') "# Set MD cell and read pdb coordinates "
  write(11,'(A)') "# "
  write(11,'(A)') "&SETUP "
  write(11,'(A10,f9.2)') "   CRYSTAL", boxl
  write(11,'(A)') "&END "


!-------------------- REM SETUP NAMELIST --------------------------------------

  write(16,'(A)') "&SETUP                                                                  "
  write(16,'(A)') "   CRYSTAL  lxcell "
  write(16,'(A)') "   READ_PDB ../urem_start.pdb                                     "
  write(16,'(A)') "&END                                                                      "

!--------------------SOLVENT NAMELIST --------------------------------------

  lx=nint(boxl/2.90)
  write(12,'(A)') "&SOLUTE                                                                   "
  write(12,'(A)') "   COORDINATES startu.pdb  "
  write(12,'(A)') "&END                                                                      "
  write(12,'(A)') "&SOLVENT                                                                  "
  write(12,'(A13,3i4)') "   GENERATE ",lx,lx,lx
  write(12,'(A)') "   CELL  SC                                                               "
  write(12,'(A)') "   INSERT 1.4                                                             "
  write(12,'(A)') "   COORDINATES ORAC_HOME/pdb/water.pdb "
  write(12,'(A)') "&END                                                                      "


!-------------------- REM SOLVENT NAMELIST --------------------------------------

  write(17,'(A)') "&SOLVENT                                                                  "
  write(17,'(A)') "   ADD_UNITS nmolslv "
  write(17,'(A)') "&END                                                                      "

  dd=coml
  dist=dsqrt(dd(1)**2+dd(2)**2+dd(3)**2)
  pme(1)=lx*3
  pme(2)=lx*3
  pme(3)=lx*3
!  write(6,*) pme
  do i=1,3
     if(pme(i).gt.64) pme(i)=64
     if(pme(i).lt.64.and.pme(i).ge.48) pme(i)=48
     if(pme(i).lt.48.and.pme(i).gt.40) pme(i)=40
     if(pme(i).lt.40.and.pme(i).ge.23) pme(i)=32
     if(pme(i).lt.32.and.pme(i).gt.24) pme(i)=24
     if(pme(i).lt.24.and.pme(i).ge.16) pme(i)=16
     if(pme(i).lt.16) pme(i)=16
  end do

!--------------------POTENTIAL  NAMELIST --------------------------------------
  write(13,'(A)') "&POTENTIAL                                                                "
  write(13,'(3x,A15,3i4,A4)') "EWALD PME 0.37  ", pme(1),pme(2),pme(3),"   4"
  write(13,'(A)') "   UPDATE      60.0   1.8                                                 "
  write(13,'(A)') "   STRETCHING HEAVY"
  write(13,'(A)') "   QQ-FUDGE  0.83333                                                      "
  write(13,'(A)') "   LJ-FUDGE  0.50                                                         "
  write(13,'(A)') "&END                                                                      "


!--------------------PROPERTIES  NAMELIST --------------------------------------
  write(18,'(A)') "&PROPERTIES                                                                "
  write(18,'(A23,i4,i10)')  "      DEF_FRAGMENT   1 ",nl
  write(18,'(A)') "&END                                                                      "

!-----------------REM NAMELIST ----------------------------------------------

  write(15,'(A)') "&REM             "
  write(15,'(A)') "   BATTERIES 8"
  write(15,'(A)') "   SETUP    1.0           0.1         0.1         1"
  write(15,'(A)') "   STEP 90.0"
  write(15,'(A)') "   PRINT_DIAGNOSTIC 900."
  write(15,'(A)') "   SEGMENT   "
  i=1
  write(15,'(6x,A6,2i6,a5)') "define",i,nl
  write(15,'(A)')  "      kind intra"  
  write(15,'(A)')  "  END "
  write(15,'(A)')  "  PRINT 18000.0 "
  write(15,'(A)')  "  PRINT_ENERGY 9000.0 OPEN 1.rem"
  write(15,'(A)') "&END                                          "

  

end program volume


subroutine inertia(mass,r,n,t) 
  implicit none
  integer n
  real*8 mass(n),r(3,n),t(3,3)
  !local 
  real*8 rr(n),delta(3,3)
  integer i,j,k
  rr=0.d0
  delta=0
  do i=1,3
     delta(i,i)=1.D0
  end do
  do k=1,n
     do j=1,3
        rr(k)=rr(k)+r(j,k)**2
     end do
  end do
  t=0.d0
  do i=1,3
     do j=i,3
        do k=1,n
           t(i,j) = t(i,j) + mass(k)*( delta(i,j)*rr(k) - r(i,k)*r(j,k) )
        end do
     end do
  end do
  do i=1,3
     do j=i,3
           t(j,i) = t(i,j)
     end do
  end do

end subroutine inertia

subroutine rotate(x,rot)
  implicit none
  REAL*8 x(3),new(3),ROT(3,3),A,B,C
  INTEGER I,K
  DO I=1,3
     NEW(i)=0.d0
     DO K=1,3
        NEW(I)=NEW(I)+ROT(k,i)*x(k)
     END DO
  end do
  x=new  ! overwrite old x
end subroutine rotate

subroutine get_mass(atom,mass) 
  implicit none
  character*4 atom,batom
  real*8 mass
  batom=adjustl(atom)
  if(batom(1:1).eq."C") mass=12.01 
  if(batom(1:1).eq."N") mass=14.01 
  if(batom(1:1).eq."O") mass=16.0 
  if(batom(1:1).eq."H") mass=1.008 
  if(batom(1:1).eq."F") mass=19.0 
  if(batom(1:1).eq."I") mass=126.90 
  if(batom(1:1).eq."P") mass=30.97 
  if(batom(1:2).eq."CL") mass=35.45 
  if(batom(1:2).eq."BR") mass=79.90 
  if(batom(1:1).eq."S") mass=32.06 

  if(batom(1:1).eq."c") mass=12.01 
  if(batom(1:1).eq."n") mass=14.01 
  if(batom(1:1).eq."o") mass=16.0 
  if(batom(1:1).eq."h") mass=1.008 
  if(batom(1:1).eq."f") mass=19.0 
  if(batom(1:1).eq."i") mass=126.90 
  if(batom(1:1).eq."p") mass=30.97 
  if(batom(1:2).eq."cl") mass=35.45 
  if(batom(1:2).eq."br") mass=79.90 
  if(batom(1:1).eq."s") mass=32.06 
end subroutine get_mass

subroutine get_type(mass,atom) 
  implicit none
  character*4 atom
  real*8 mass
  if(mass.eq.12.0 )  atom="C"
  if(mass.eq.14.0 )  atom="N"
  if(mass.eq.16.0 )  atom="O"
  if(mass.eq.1.0 )   atom="H"
  if(mass.eq.19.0 )  atom="F"
  if(mass.eq.126.0 ) atom="I"
  if(mass.eq.31.0 )  atom="P"
  if( mass.eq.35.0 ) atom="CL"
  if(mass.eq.79.0 )  atom="BR"
end subroutine get_type


subroutine Jacobi(a,x,abserr,n)
!===========================================================
! Evaluate eigenvalues and eigenvectors
! of a real symmetric matrix a(n,n): a*x = lambda*x 
! method: Jacoby method for symmetric matrices 
! Alex G. (December 2009)
!-----------------------------------------------------------
! input ...
! a(n,n) - array of coefficients for matrix A
! n      - number of equations
! abserr - abs tolerance [sum of (off-diagonal elements)^2]
! output ...
! a(i,i) - eigenvalues
! x(i,j) - eigenvectors
! comments ...
!===========================================================
implicit none
integer i, j, k, n
double precision a(n,n),x(n,n),arr(n)
double precision abserr, b2, bar
double precision beta, coeff, c, s, cs, sc

! initialize x(i,j)=0, x(i,i)=1
! *** the array operation x=0.0 is specific for Fortran 90/95
x = 0.0
do i=1,n
  x(i,i) = 1.0
end do

! find the sum of all off-diagonal elements (squared)
b2 = 0.0
do i=1,n
  do j=1,n
    if (i.ne.j) b2 = b2 + a(i,j)**2
  end do
end do

if (b2 <= abserr) return

! average for off-diagonal elements /2
bar = 0.5*b2/float(n*n)

do while (b2.gt.abserr)
  do i=1,n-1
    do j=i+1,n
      if (a(j,i)**2 <= bar) cycle  ! do not touch small elements
      b2 = b2 - 2.0*a(j,i)**2
      bar = 0.5*b2/float(n*n)
! calculate coefficient c and s for Givens matrix
      beta = (a(j,j)-a(i,i))/(2.0*a(j,i))
      coeff = 0.5*beta/sqrt(1.0+beta**2)
      s = sqrt(max(0.5+coeff,0.0))
      c = sqrt(max(0.5-coeff,0.0))
! recalculate rows i and j
      do k=1,n
        cs =  c*a(i,k)+s*a(j,k)
        sc = -s*a(i,k)+c*a(j,k)
        a(i,k) = cs
        a(j,k) = sc
      end do
! new matrix a_{k+1} from a_{k}, and eigenvectors 
      do k=1,n
        cs =  c*a(k,i)+s*a(k,j)
        sc = -s*a(k,i)+c*a(k,j)
        a(k,i) = cs
        a(k,j) = sc
        cs =  c*x(k,i)+s*x(k,j)
        sc = -s*x(k,i)+c*x(k,j)
        x(k,i) = cs
        x(k,j) = sc
      end do
    end do
  end do
end do
! now sort 
do i=1,n
   arr(i)=a(i,i) 
end do
call piksr2(n,arr,x)
do i=1,n
   a(i,i)=arr(i)
end do
return
end
SUBROUTINE piksr2(n,arr,brr)
  implicit none
  INTEGER n
  REAL*8 arr(n),brr(n,n)
  !Sorts an array arr(1:n) into ascending numerical order, by straight insertion, while making
  !the corresponding rearrangement of the array brr(1:n,*).
  INTEGER i,j,k
  REAL*8 a,b(n)
  do j=2,n ! Pick out each element in turn.
     a=arr(j)
     do k=1,n
        b(k)=brr(k,j)
     end do
     do i=j-1,1,-1 ! Look for the place to insert it.
        if(arr(i).le.a) goto 10
        arr(i+1)=arr(i)
        do k=1,n
           brr(k,i+1)=brr(k,i)
        end do
     enddo
     i=0
10   arr(i+1)=a ! Insert it.
     do k=1,n
        brr(k,i+1)=b(k)
     end do
  enddo
  return
END SUBROUTINE piksr2


subroutine side_chain(i,residue,istart,iend)
  integer i,istart,iend
  character*5 residue
  if(trim(adjustl(residue)).EQ."ARG") THEN
     istart=i+4
     iend=i+21
  end if
  if(trim(adjustl(residue)).EQ."LYS") THEN
     istart=i+4
     iend=i+19
  end if
  if(trim(adjustl(residue)).EQ."ASP") THEN
     istart=i+4
     iend=i+9
  end if
  if(trim(adjustl(residue)).EQ."GLU") THEN
     istart=i+4
     iend=i+12
  end if
  if(trim(adjustl(residue)).EQ."TYR") THEN
     istart=i+4
     iend=i+18
  end if
  if(trim(adjustl(residue)).EQ."THR") THEN
     istart=i+4
     iend=i+11
  end if
  if(trim(adjustl(residue)).EQ."ILE") THEN
     istart=i+4
     iend=i+16
  end if
  if(trim(adjustl(residue)).EQ."LEU") THEN
     istart=i+4
     iend=i+16
  end if
  if(trim(adjustl(residue)).EQ."HSD") THEN
     istart=i+4
     iend=i+14
  end if
  if(trim(adjustl(residue)).EQ."HSP") THEN
     istart=i+4
     iend=i+15
  end if
  if(trim(adjustl(residue)).EQ."SER") THEN
     istart=i+4
     iend=i+8
  end if
  if(trim(adjustl(residue)).EQ."ASN") THEN
     istart=i+4
     iend=i+11
  end if
  if(trim(adjustl(residue)).EQ."ALA") THEN
     istart=i+4
     iend=i+7
  end if
  if(trim(adjustl(residue)).EQ."VAL") THEN
     istart=i+4
     iend=i+13
  end if
  if(trim(adjustl(residue)).EQ."PHE") THEN
     istart=i+4
     iend=i+17
  end if
  if(trim(adjustl(residue)).EQ."GLN") THEN
     istart=i+4
     iend=i+14
  end if
  if(trim(adjustl(residue)).EQ."TRP") THEN
     istart=i+4
     iend=i+21
  end if
  if(trim(adjustl(residue)).EQ."MET") THEN
     istart=i+4
     iend=i+14
  end if
  return
end subroutine side_chain




