!==========================================================================
program rms
!==========================================================================
!     Fast rms routine form proteins.
!     Computes the deviation from a reference structure comparing 
!     two pdb files. Written by Procacci CECAM 
!     WARNING !!!!!!!!!!!!!
!     WARNING 
!     WARNING  : on Linux compile with -finit-local-zero -fno-automatic
!     WARNING 
!     WARNING !!!!!!!!!!!!!!!!!!!!!
!    syntax: rms sub1 full1 sub2 full2 
!          where  sub0 =  pdb file of reference structure (subset)  
!                 full0 = pdb file of reference structure  (full) 
!                 sub1 =  pdb file of moved  structure (subset)  
!                 full1 = pdb file of moved  structure  (full) 
!          sub0 and sub1 must contain the same number of atoms 
!          given in identical order. 
!==========================================================================

  implicit  none

  integer    nmax,mat
  parameter (mat=50000)


  !--------------------------------------------------------------------------
  !     r0 = coordinates of the reference structure 
  !     r  = coordinates of the displaced structure 
  !     rr = coordinates of the closest rotated and traslated structure 
  !     rio = rotation matrix 
  !     p = angles of rotation  (radiants) 
  !     t = t matrix (see comment below) 
  !     delta = 3d kroenecker delta      
  !--------------------------------------------------------------------------

  real*8 :: r(3,mat),rq(3,mat),r0(3,mat),rr(3,mat),t_mat(3,3),p(3),rio(3,3)
  real*8 :: riom(3,3),delta(3,3),r0tot,rtot,x,y,z,xi(3,3),ftol,fret
  real*8 :: fretmin,dd,pi,dumm,tcpu,cpu,r0_t(3),r_t(3),r0_all(3,mat)
  real*8 :: r_all(3,mat),rr_all(3,mat),mean
  character*4 ath(mat)
  character*30 ath_all(mat),athr_all(mat)
  character*1 at1
  character*2 at2
  character*30 at  
  integer i,j,k,l,m,n,k0,i1,j1,n3,iter,k0all,kall
  common /all_i_need/t_mat,rio
  data  delta /1.0,0.0,0.0,0.0,1.0,0.0,0.0,0.0,1.0/

  !     open input files 
  call opens

  k0=0
  n=100000
  !     read reference pdb file  (alpha carbon only) 
  do i=1,n 
     read(9,10,end=1000) at,x,y,z
10   format(a30,3f8.3,10x) 
     at2=at(13:14)
     call Stripspaces(at2)           
     at1 = at2
     if(at1.ne."H".and.at1.ne."h") then 
        k0=k0+1
        ath(k0)=at(13:16)
        r0(1,k0)=x 
        r0(2,k0)=y 
        r0(3,k0)=z
     end if
  end do
1000 continue

  k0all=0
  n=100000
  !     read reference pdb file (full) 
  do i=1,n 
     read(10,10,end=1001) at,x,y,z
     at2=at(13:14)
     call Stripspaces(at2)           
     at1 = at2
     if(at1.ne."H".and.at1.ne."h") then 
        k0all=k0all+1
        ath_all(k0all)=at
        r0_all(1,k0all)=x 
        r0_all(2,k0all)=y 
        r0_all(3,k0all)=z
     end if
  end do
1001 continue



  k=0
  !     read pdb file of the displaced protein (alpha carbon only)  
  do i=1,n 
     read(11,10,end=2000) at,x,y,z
     at2=at(13:14)
     call Stripspaces(at2)           
     at1 = at2
     if(at1.ne."H".and.at1.ne."h") then 
        k=k+1
        r(1,k)=x 
        r(2,k)=y 
        r(3,k)=z
     end if
  end do

2000 CONTINUE

  kall=0
  !     read pdb file of the displaced protein  (full)
  do i=1,n 
     read(12,10,end=2001) at,x,y,z
     at2=at(13:14)
     call Stripspaces(at2)           
     at1 = at2
     if(at1.ne."H".and.at1.ne."h") then 
        kall=kall+1
        athr_all(kall)=at
        r_all(1,kall)=x 
        r_all(2,kall)=y 
        r_all(3,kall)=z
     end if
  end do
2001 CONTINUE


  !     stop if n of atoms is not the same 
  if(k.ne.k0) then 
     write(6,11) k0,k
11   format(/' number of heavy atoms does not match for', &
          'the two structure: cannot go any further',/ &
          ' n (ref) = ',i5, '   n(moved) =', i5)
     stop
  end if
  n=k 

  write(6,3001) k0,k
3001 format(" ---> n. of atoms of ref. and moved structure (subset) = ",i5,i5)     
  !     shift origin to centroid position for both struct. 
  do j=1,3
     r0tot = 0.d0
     rtot = 0.d0
     do i=1,n
        r0tot = r0tot+r0(j,i)
        rtot = rtot+r(j,i)
     end do
     r0_t(j) = r0tot/float(n) 
     r_t(j) = rtot/float(n) 

     do i=1,n 
        r0(j,i)=r0(j,i)- r0tot/float(n)
        r(j,i)=r(j,i)- rtot/float(n)
     end do
     do i=1,k0all
        r0_all(j,i)= r0_all(j,i) -  r0tot/float(n)
     end do
     do i=1,kall
        r_all(j,i)= r_all(j,i) -  rtot/float(n)
     end do
  end do

  !     find out "best" rotation matrix between the two structure
  !     maximazing the quantity T = sum_{ij} T_{ij} * R_{ij}, where
  !     T_{ij}= sum_{i}= r0(i)r(j) and R is a rotation matrix    
  !     initial guess for R  is identity. Extrum search is done
  !     with numerical recipes Powell routine   

  !     computes the T matrix once for all and set R to I 

  do i=1,3
     do j=1,3
        xi(i,j) = delta(i,j)
        t_mat(i,j)=0.d0
        do k=1,n
           t_mat(i,j) = t_mat(i,j) + r0(i,k)*r(j,k)
        end do
     end do
  end do
  p(1) = 0.0
  p(2) = 0.0
  p(3) = 0.0

  !     start minimization using Powell
  !     initial guess is done on a coarse grid to avoid trapping 
  !     in local minima.        

  ftol = 0.00001
  fretmin = 1d10
  pi =dacos(-1.d0)
  n3=3
  ITER =50
  CALL timer(dumm,tcpu,dumm)
  cpu=tcpu
  do i=1,2
     p(1) = p(1)+0.3333*pi
     p(2) = 0.d0
     do j=1,2
        p(2) = p(2)+0.3333*pi/2 
        p(3) = 0.d0
        do k=1,2
           p(3) = p(3)+0.3333*pi/2
           call powell(p,XI,N3,N3,FTOL,ITER,FRET)
           if (fret.lt.fretmin) then 
              fretmin = fret 
              do i1=1,3
                 do j1=1,3
                    riom(i1,j1) = rio(i1,j1) 
                 end do
              end do
           end if
        end do
     end do
  end do

  !     time for minimization is less than 0.002 seconds on DEC-alpha
  !     irrespetively of the number of atoms.  

  !     computes "closest" coordinates 

  write(6,1003)
1003 format(" ---> Rotation matrix")
  do i=1,3
     write(6,1002) (riom(i,j),j=1,3)
1002 format(10x,3f10.5)
  end do

  CALL timer(dumm,tcpu,dumm)
  cpu=-cpu+tcpu
  write(6,1005) cpu
1005 format("--> CPU Time for minimization =",g15.5," seconds") 
  do i=1,3
     do l = 1,n
        rtot = 0.d0
        do j =1,3
           rtot = rtot + riom(i,j)*r(j,l)
        end do
        rr(i,l) = rtot 
     end do
  end do

  do i=1,3
     do l = 1,kall
        rtot = 0.d0
        do j =1,3
           rtot = rtot + riom(i,j)*r_all(j,l)
        end do
        rr_all(i,l) = rtot 
     end do
  end do

  open(unit=13,file="reference.pdb",form="formatted") 
  open(unit=14,file="rotated.pdb",form="formatted") 
  do i=1,k0all 
     write(13,10) ath_all(i),r0_all(1,i),r0_all(2,i),r0_all(3,i)
  end do

  do i=1,kall 
     write(14,10) athr_all(i),rr_all(1,i),rr_all(2,i),rr_all(3,i)
  end do


  !     print out rms 

  write(6,1006)
1006 format(" ---> RMSs ",/, 7x," Ref struct",14x," Rotated struct",10x,"RMS"/ )
  write(6,*) 
  mean=0
  do l=1,n
     dd = 0.0
     do j=1,3
        dd = dd+ (r0(j,l)-rr(j,l))**2
     end do
     write(6,19) l,ath(l),r0(1,l),r0(2,l),r0(3,l),rr(1,l),rr(2,l),rr(3,l),dsqrt(dd)
19   format(i5,2x,a4,3f8.4,5x,3f8.4,f10.4) 
     mean= mean +dsqrt(dd)
  end do
  mean=mean/float(n)
  write(6,29) mean 
29 format(" meanRMS = ", f10.6)

  !!    now do computation with procacci's method. 
  stop
  riom=0.d0



!==================== EULER ROTATION MATRIX ==================================

  call rigid(n,r0,r,riom) 


  do i=1,3
     do l = 1,kall
        rtot = 0.d0
        do j =1,3
           rtot = rtot + riom(i,j)*r_all(j,l)
        end do
        rr_all(i,l) = rtot 
     end do
  end do

  open(unit=15,file="ref_proca.pdb",form="formatted") 
  open(unit=16,file="rot_proca.pdb",form="formatted") 
  do i=1,k0all 
     write(15,10) ath_all(i),r0_all(1,i),r0_all(2,i),r0_all(3,i)
  end do

  do i=1,kall 
     write(16,10) athr_all(i),rr_all(1,i),rr_all(2,i),rr_all(3,i)
  end do

  !     print out rms 

  write(6,*) 
  do l=1,n
     dd = 0.0
     do j=1,3
        dd = dd+ (r0(j,l)-r(j,l))**2
     end do
     write(6,19) l,ath(l),r0(1,l),r0(2,l),r0(3,l),r(1,l),r(2,l),r(3,l),dsqrt(dd)
  end do


!==================== QUATERNION MATRIX ==================================

  do i=1,20
     riom=0.d0
     call rigidq(n,r0,r,riom) 
  end do
  
  do i=1,3
     do l = 1,kall
        rtot = 0.d0
        do j =1,3
           rtot = rtot + riom(i,j)*r_all(j,l)
        end do
        rr_all(i,l) = rtot 
     end do
  end do

  open(unit=15,file="Qref_proca.pdb",form="formatted") 
  open(unit=16,file="Qrot_proca.pdb",form="formatted") 
  do i=1,k0all 
     write(17,10) ath_all(i),r0_all(1,i),r0_all(2,i),r0_all(3,i)
  end do

  do i=1,kall 
     write(18,10) athr_all(i),rr_all(1,i),rr_all(2,i),rr_all(3,i)
  end do

  !     print out rms 

  write(6,*) 
  do l=1,n
     dd = 0.0
     do j=1,3
        dd = dd+ (r0(j,l)-r(j,l))**2
     end do
     write(6,19) l,ath(l),r0(1,l),r0(2,l),r0(3,l),r(1,l),r(2,l),r(3,l),dsqrt(dd)
  end do


  stop 
end program rms

function func(p) 
  implicit none 
  real*8   p(*),delta(3,3), rx(3,3),ry(3,3),rz(3,3),rio(3,3,3),norm(3),t_mat(3,3),func,xmin,riomin(3,3) 
  common /all_i_need/t_mat,riomin
  data   delta /1.0,0.0,0.0,0.0,1.0,0.0,0.0,0.0,1.0/
  integer  i,j,k,l,m,n
  
  do i=1,3
     do j=i,3
        rx(i,j)=delta(i,j)
        ry(i,j)=delta(i,j)
        rz(i,j)=delta(i,j)
     end do
  end do
  
  
  rx(2,2)=cos(p(1))
  rx(2,3)=sin(p(1))
  rx(3,3)=cos(p(1))
  rx(3,2)=-sin(p(1))
  ry(1,1)=cos(p(2))
  ry(1,3)=sin(p(2))
  ry(3,3)=cos(p(2))
  ry(3,1)=-sin(p(2))
  rz(1,1)=cos(p(3))
  rz(1,2)=sin(p(3))
  rz(2,2)=cos(p(3))
  rz(2,1)=-sin(p(3))

!     build up rotation matrices

  do i=1,3
     do j=1,3
        rio(1,i,j) = 0.d0
        rio(2,i,j) = 0.d0
        rio(3,i,j) = 0.d0
        do k=1,3
           do l=1,3
              rio(1,i,j) = rio(1,i,j) + rx(i,k)*ry(k,l)*rz(l,j)
              rio(2,i,j) = rio(2,i,j) + ry(i,k)*rx(k,l)*rz(l,j)
              rio(3,i,j) = rio(3,i,j) + rz(i,k)*ry(k,l)*rx(l,j)
           end do
        end do
     end do
  end do

  xmin = 1.0d44
  
  do k=1,3
     norm(k) = 0.d0
     do i=1,3
        do j=1,3
           norm(k) = norm(k) - t_mat(i,j)*rio(k,i,j) 
        end do
     end do
     if(norm(k).lt.xmin) then 
        xmin = norm(k)
        do i=1,3
           do j=1,3
              riomin(i,j)=rio(k,i,j)
           end do
        end do
     end if
  end do
  func = xmin 
  return
end function func

SUBROUTINE POWELL(P,XI,N,NP,FTOL,ITER,FRET)
  implicit real*8 (a-h,o-z)
  PARAMETER (NMAX=30,ITMAX=100)
  DIMENSION P(NP),XI(NP,NP),PT(NMAX),PTT(NMAX),XIT(NMAX)
  FRET=FUNC(P)
  DO  J=1,N
     PT(J)=P(J)
  end do
  ITER=0
1 ITER=ITER+1
  FP=FRET
  if(fret.ge.1.d25) return
  IBIG=0
  DEL=0.
  DO I=1,N
     DO  J=1,N
        XIT(J)=XI(J,I)
     end do
     CALL LINMIN(P,XIT,N,FRET)
     IF(ABS(FP-FRET).GT.DEL)THEN
        DEL=ABS(FP-FRET)
        IBIG=I
     ENDIF
  end do
  IF(2.*ABS(FP-FRET).LE.FTOL*(ABS(FP)+ABS(FRET)))RETURN
  IF(ITER.EQ.ITMAX) then
     write (6,*) 'Powell exceeding maximum iterations.'
     fret=2.d25
     return
  endif
  DO J=1,N
     PTT(J)=2.*P(J)-PT(J)
     XIT(J)=P(J)-PT(J)
     PT(J)=P(J)
  end do
  FPTT=FUNC(PTT)
  IF(FPTT.GE.FP)GO TO 1
  T=2.*(FP-2.*FRET+FPTT)*(FP-FRET-DEL)**2-DEL*(FP-FPTT)**2
  IF(T.GE.0.)GO TO 1
  CALL LINMIN(P,XIT,N,FRET)
  DO  J=1,N
     XI(J,IBIG)=XIT(J)
  end do
  GO TO 1
END SUBROUTINE POWELL

SUBROUTINE LINMIN(P,XI,N,FRET)                                            
  implicit real*8 (a-h,o-z)
  PARAMETER (NMAX=30,TOL=1.E-4)                                             
  EXTERNAL F1DIM                                                            
  DIMENSION P(N),XI(N)                                                      
  COMMON /F1COM/ PCOM(NMAX),XICOM(NMAX),NCOM                                
  NCOM=N                                                                    
  DO  J=1,N                                                               
     PCOM(J)=P(J)                                                            
     XICOM(J)=XI(J)                                                          
  end do
  AX=0.                                                                     
  XX=1.                                                                     
  BX=2.                                                                     
  CALL MNBRAK(AX,XX,BX,FA,FX,FB,F1DIM)                                      
  FRET=BRENT(AX,XX,BX,F1DIM,TOL,XMIN)                                       
  DO  J=1,N                                                               
     XI(J)=XMIN*XI(J)                                                        
     P(J)=P(J)+XI(J)                                                         
  end DO
  RETURN                                                                    
END SUBROUTINE LINMIN

SUBROUTINE MNBRAK(AX,BX,CX,FA,FB,FC,FUNC)                                 
  implicit real*8 (a-h,o-z)
  PARAMETER (GOLD=1.618034, GLIMIT=100., TINY=1.E-20)                       
  FA=FUNC(AX)                                                               
  FB=FUNC(BX)                                                               
  IF(FB.GT.FA)THEN                                                          
     DUM=AX                                                                  
     AX=BX                                                                   
     BX=DUM                                                                  
     DUM=FB                                                                  
     FB=FA                                                                   
     FA=DUM                                                                  
  ENDIF
  CX=BX+GOLD*(BX-AX)                                                        
  FC=FUNC(CX)                                                               
1 IF(FB.GE.FC)THEN                                                          
     R=(BX-AX)*(FB-FC)                                                       
     Q=(BX-CX)*(FB-FA)                                                       
     U=BX-((BX-CX)*Q-(BX-AX)*R)/(2.*SIGN(MAX(ABS(Q-R),TINY),Q-R))            
     ULIM=BX+GLIMIT*(CX-BX)                                                  
     IF((BX-U)*(U-CX).GT.0.)THEN                                             
        FU=FUNC(U)                                                            
        IF(FU.LT.FC)THEN                                                      
           AX=BX                                                               
           FA=FB                                                               
           BX=U                                                                
           FB=FU                                                               
           GO TO 1                                                             
        ELSE IF(FU.GT.FB)THEN                                                 
           CX=U                                                                
           FC=FU                                                               
           GO TO 1                                                             
        ENDIF
        U=CX+GOLD*(CX-BX)                                                     
        FU=FUNC(U)                                                            
     ELSE IF((CX-U)*(U-ULIM).GT.0.)THEN                                      
        FU=FUNC(U)                                                            
        IF(FU.LT.FC)THEN                                                      
           BX=CX                                                               
           CX=U                                                                
           U=CX+GOLD*(CX-BX)                                                   
           FB=FC                                                               
           FC=FU                                                               
           FU=FUNC(U)                                                          
        ENDIF
     ELSE IF((U-ULIM)*(ULIM-CX).GE.0.)THEN                                   
        U=ULIM                                                                
        FU=FUNC(U)                                                            
     ELSE                                                                    
        U=CX+GOLD*(CX-BX)                                                     
        FU=FUNC(U)                                                            
     ENDIF
     AX=BX                                                                   
     BX=CX                                                                   
     CX=U                                                                    
     FA=FB                                                                   
     FB=FC                                                                   
     FC=FU                                                                   
     GO TO 1                                                                 
  ENDIF
  RETURN                                                                    
END SUBROUTINE MNBRAK

FUNCTION F1DIM(X)                                                         
  implicit real*8 (a-h,o-z)
  PARAMETER (NMAX=30)                                                       
  COMMON /F1COM/ PCOM(NMAX),XICOM(NMAX),NCOM                                
  DIMENSION XT(NMAX)                                                        
  DO  J=1,NCOM                                                            
     XT(J)=PCOM(J)+X*XICOM(J)                                                
  end do
  F1DIM=FUNC(XT)                                                            
  RETURN                                                                    
END FUNCTION F1DIM

FUNCTION BRENT(AX,BX,CX,F,TOL,XMIN)                                       
  implicit real*8 (a-h,o-z)
  PARAMETER (ITMAX=100,CGOLD=.3819660,ZEPS=1.0E-10)                         
  A=MIN(AX,CX)                                                              
  B=MAX(AX,CX)                                                              
  V=BX                                                                      
  W=V                                                                       
  X=V                                                                       
  E=0.                                                                      
  FX=F(X)                                                                   
  FV=FX                                                                     
      FW=FX                                                                     
      DO  ITER=1,ITMAX                                                        
         XM=0.5*(A+B)                                                            
        TOL1=TOL*ABS(X)+ZEPS                                                    
        TOL2=2.*TOL1                                                            
        IF(ABS(X-XM).LE.(TOL2-.5*(B-A))) GOTO 3                                 
        IF(ABS(E).GT.TOL1) THEN                                                 
           R=(X-W)*(FX-FV)                                                       
           Q=(X-V)*(FX-FW)                                                       
           P=(X-V)*Q-(X-W)*R                                                     
           Q=2.*(Q-R)                                                            
           IF(Q.GT.0.) P=-P                                                      
           Q=ABS(Q)                                                              
           ETEMP=E                                                               
           E=D                                                                   
           IF(ABS(P).GE.ABS(.5*Q*ETEMP).OR.P.LE.Q*(A-X).OR.P.GE.Q*(B-X)) GOTO 1                                              
           D=P/Q                                                                 
           U=X+D                                                                 
           IF(U-A.LT.TOL2 .OR. B-U.LT.TOL2) D=SIGN(TOL1,XM-X)                    
           GOTO 2                                                                
        ENDIF
1       IF(X.GE.XM) THEN                                                        
           E=A-X                                                                 
        ELSE                                                                    
           E=B-X                                                                 
        ENDIF
        D=CGOLD*E                                                               
2       IF(ABS(D).GE.TOL1) THEN                                                 
           U=X+D                                                                 
        ELSE                                                                    
           U=X+SIGN(TOL1,D)                                                      
        ENDIF
        FU=F(U)                                                                 
        IF(FU.LE.FX) THEN                                                       
           IF(U.GE.X) THEN                                                       
              A=X                                                                 
           ELSE                                                                  
              B=X                                                                 
           ENDIF
           V=W                                                                   
           FV=FW                                                                 
           W=X                                                                   
           FW=FX                                                                 
           X=U                                                                   
           FX=FU                                                                 
        ELSE                                                                    
           IF(U.LT.X) THEN                                                       
              A=U                                                                 
           ELSE                                                                  
              B=U                                                                 
           ENDIF
           IF(FU.LE.FW .OR. W.EQ.X) THEN                                         
              V=W                                                                 
              FV=FW                                                               
              W=U                                                                 
              FW=FU                                                               
           ELSE IF(FU.LE.FV .OR. V.EQ.X .OR. V.EQ.W) THEN                        
              V=U                                                                 
              FV=FU                                                               
           ENDIF
        ENDIF
     end do
     write(6,*) 'Brent exceed maximum iterations.' 
     STOP
3    XMIN=X                                                                    
     BRENT=FX                                                                  
     RETURN                                                                    
   END FUNCTION BRENT


   SUBROUTINE OPENS 
     !  Opens parms passed to command as unis 7, 8, ...
     PARAMETER( MAXLEN = 80 )
     CHARACTER*80 STRING
     N = 1
1    N = N + 1
     CALL GETPRM( N-1, STRING, LENGHT )
     IF( LENGHT .EQ. 0) THEN 
        RETURN
     ENDIF
     WRITE(*,*) ' Fortran unit ', N+7, ' ', STRING(1:LENGHT)
     OPEN( UNIT=N+7, FILE=STRING(1:LENGHT),STATUS='UNKNOWN' )
     REWIND( N+7 )
     GOTO 1
   END SUBROUTINE OPENS
   
   SUBROUTINE BLSTRIP( STRING, LENGHT )
     !  Strip trailing blanks from STRING, returns LENGTH
     PARAMETER( MAXLEN = 80 )
     CHARACTER*80 STRING
     
     LENGHT = MAXLEN+1
1    LENGHT = LENGHT - 1
     IF( LENGHT .EQ. 0 ) RETURN
     IF( STRING(LENGHT:LENGHT) .EQ. ' ' ) GOTO 1
     
     RETURN
   END SUBROUTINE BLSTRIP
   
   SUBROUTINE GETPRM( N, STRING, LENGHT )
     !  Returns N-th parameter on the command line (if LENGHT .gt. 0)
     PARAMETER( MAXLEN = 80 )
     CHARACTER*80 STRING
     
     CALL GETARG( N, STRING )
     CALL BLSTRIP( STRING, LENGHT )
     
     RETURN
   END SUBROUTINE GETPRM
   
   SUBROUTINE timer(vfcp,tfcp,elapse)
     REAL*8 t2(2),elapse,vfcp,tfcp
     !      tfcp=etime(t2)
     !      vfcp=t2(1)
     !      elapse=secnds(0.0) 
     RETURN
   END SUBROUTINE timer
   

subroutine rigid(nato,r0,r,rio)
  !args 
  implicit none
  real*8 :: r0(3,nato),r(3,nato), rio(3,3),rioq(3,3) ! com relative rigid r0 and distorted coordinates.
  integer nato
  !local 
  real*8 t_mat(3,3), ts_mat(3,3), tinv(3,3), tq_mat(3,3), tinv_q(3,3)   ! T and Ts matrix  and Inverse of Ts 
  real*8 tvec(3),tqvec(3),tt(3),ttq(3),ut(3),utq(3),t ,tq,ctheta,stheta,ctheta1,cthetaq,sthetaq,cthetaq1,rtot,xx,det,pi
  integer i,j,k,l

  ! compute ts_mat
  do i=1,3
     do j=1,3
        t_mat(i,j)=0.d0
        do k=1,nato
           t_mat(i,j) = t_mat(i,j) + r0(i,k)*r(j,k)
        end do
     end do
  end do
  do i=1,3
     do j=i,3
        ts_mat(i,j)=t_mat(i,j)+t_mat(j,i) 
        ts_mat(j,i)=ts_mat(i,j)
     end do
  end do

  !   compute T vec

  tvec(1)=t_mat(3,2)-t_mat(2,3)
  tvec(2)=t_mat(1,3)-t_mat(3,1)
  tvec(3)=t_mat(2,1)-t_mat(1,2)

  !   compute inverse 
  det = ts_mat(1,1)*( ts_mat(3,3)*ts_mat(2,2)- ts_mat(3,2)*ts_mat(2,3)) - &
       ts_mat(2,1)*( ts_mat(3,3)*ts_mat(1,2)- ts_mat(3,2)*ts_mat(1,3) ) + & 
       ts_mat(3,1)*( ts_mat(2,3)*ts_mat(1,2)- ts_mat(2,2)*ts_mat(1,3) ) 
  tinv(1,1)=    ts_mat(3,3)*ts_mat(2,2)- ts_mat(3,2)*ts_mat(2,3)
  tinv(1,2)= -( ts_mat(3,3)*ts_mat(1,2)- ts_mat(3,2)*ts_mat(1,3) )
  tinv(1,3)=    ts_mat(2,3)*ts_mat(1,2)- ts_mat(2,2)*ts_mat(1,3)
  tinv(2,2)=    ts_mat(3,3)*ts_mat(1,1)- ts_mat(3,1)*ts_mat(1,3)
  tinv(2,3)= -(  ts_mat(2,3)*ts_mat(1,1)- ts_mat(2,1)*ts_mat(1,3))
  tinv(3,3)=     ts_mat(2,2)*ts_mat(1,1)- ts_mat(2,1)*ts_mat(1,2)
  tinv(2,1)=tinv(1,2)
  tinv(3,1)=tinv(1,3)
  tinv(3,2)=tinv(2,3)
  
  tinv=tinv/det
  
  write(6,2103)
2103 format(" ---> Inverse  analytic matrix (Procacci)")
  do i=1,3
     write(6,1002) (tinv(i,j),j=1,3)
  end do
  do i=1,3
     do j=1,3
        xx=0.d0
        do k=1,3
           xx=xx+ts_mat(i,k)*tinv(k,j)
        end do
        write(6,*) i,j,xx
     end do
  end do

  tinv=0


  call inverse(ts_mat,tinv,3)

  ! compute t scalar

  t=0.d0
  do i=1,3
     tt(i)=0.d0
     do j=1,3
        tt(i)= tt(i)+tvec(j)*tinv(j,i)
     end do
     t = t + tt(i)**2
  end do
  write(6,2003)
2003 format(" ---> Inverse  matrix (Procacci)")
  write(6,1002) tinv

  ctheta = (1-t)/(1+t) 
  stheta = dsqrt(1-ctheta*ctheta)

  pi=dacos(-1.d0)
  write(6,*) "cos-sin",ctheta,stheta,dacos(ctheta)*180.0/pi,dasin(stheta)*180.0/pi

  ! compute ut 
  do i=1,3
     ut(i)=0.d0
     do j=1,3
        ut(i)=ut(i)+tinv(i,j)*tvec(j)
     end do
  end do

  ut=ut*stheta/(ctheta-1)
  ctheta1=1-ctheta

  write(6,*) ut,ut(1)**2+ut(2)**2+ut(3)**2.

  ! compute best rotation matrix that bring rigid r0 into distorted r
 
  rio(1,1)=ut(1)*ut(1)*ctheta1 +ctheta
  rio(2,2)=ut(2)*ut(2)*ctheta1 +ctheta
  rio(3,3)=ut(3)*ut(3)*ctheta1 +ctheta
  rio(1,2)=ut(1)*ut(2)*ctheta1 -ut(3)*stheta
  rio(2,1)=ut(1)*ut(2)*ctheta1 +ut(3)*stheta
  rio(1,3)=ut(1)*ut(3)*ctheta1 +ut(2)*stheta
  rio(3,1)=ut(1)*ut(3)*ctheta1 -ut(2)*stheta
  rio(2,3)=ut(2)*ut(3)*ctheta1 -ut(1)*stheta
  rio(3,2)=ut(2)*ut(3)*ctheta1 +ut(1)*stheta
  
  write(6,1003)
1003 format(" ---> Rotation matrix (Procacci)")
  write(6,1002) rio

1002 format(10x,3f10.5)

! From Rot matrix finds new coordinated obeying rigidity constraint

  do i=1,3
     do l = 1,nato
        rtot = 0.d0
        do j =1,3
           rtot = rtot + rio(i,j)*r0(j,l)
        end do
        r(i,l) = rtot 
     end do
  end do
  
end subroutine rigid

subroutine rigidq(nato,r0,r,rio)
  !args 
  implicit none
  real*8 :: r0(3,nato),r(3,nato), rio(3,3) ! com relative rigid r0 and distorted coordinates.
  integer nato
  !local 
  real*8 t_mat(3,3), tinv(3,3), tq_mat(3,3)   ! T and Ts matrix  and Inverse of Ts 
  real*8 tqvec(3),tt(3),ttq(3),ut(3),utq(3),t ,tq,ctheta,stheta,ctheta1,cthetaq,sthetaq,cthetaq1,rtot,xx,det,pi
  integer i,j,k,l

  ! compute ts_mat
  write(6,*) "r0", nato
  write(6,500) r0
500 format(3f10.3)
  write(6,*) "r", nato
  write(6,500) r
  do i=1,3
     do j=1,3
        t_mat(i,j)=0.d0
        do k=1,nato
           t_mat(i,j) = t_mat(i,j) + r0(i,k)*r(j,k)
        end do
     end do
  end do
  do i=1,3
     do j=i,3
        tq_mat(i,j)=t_mat(i,j)+t_mat(j,i) 
        tq_mat(j,i)=tq_mat(i,j)
     end do
  end do

  tq_mat(1,1)=-2.d0*(t_mat(2,2)+t_mat(3,3))
  tq_mat(2,2)=-2.d0*(t_mat(1,1)+t_mat(3,3))
  tq_mat(3,3)=-2.d0*(t_mat(1,1)+t_mat(2,2))

  !   compute T vec
  tqvec(1)=-(t_mat(3,2)-t_mat(2,3))
  tqvec(2)=-(t_mat(1,3)-t_mat(3,1))
  tqvec(3)=-(t_mat(2,1)-t_mat(1,2))
  write(6,*) "t_mat"
  write(6,500)  t_mat
  write(6,*) "tq_mat"
  write(6,500)  tq_mat
  write(6,*) "tvec"
  write(6,500)  tqvec

  !   compute inverse 
  det = tq_mat(1,1)*( tq_mat(3,3)*tq_mat(2,2)- tq_mat(3,2)*tq_mat(2,3)) - &
       tq_mat(2,1)*( tq_mat(3,3)*tq_mat(1,2)- tq_mat(3,2)*tq_mat(1,3) ) + & 
       tq_mat(3,1)*( tq_mat(2,3)*tq_mat(1,2)- tq_mat(2,2)*tq_mat(1,3) ) 
  write(6,*) "det",det 
  tinv(1,1)=    tq_mat(3,3)*tq_mat(2,2)- tq_mat(3,2)*tq_mat(2,3)
  tinv(1,2)= -( tq_mat(3,3)*tq_mat(1,2)- tq_mat(3,2)*tq_mat(1,3) )
  tinv(1,3)=    tq_mat(2,3)*tq_mat(1,2)- tq_mat(2,2)*tq_mat(1,3)
  tinv(2,2)=    tq_mat(3,3)*tq_mat(1,1)- tq_mat(3,1)*tq_mat(1,3)
  tinv(2,3)= -(  tq_mat(2,3)*tq_mat(1,1)- tq_mat(2,1)*tq_mat(1,3))
  tinv(3,3)=     tq_mat(2,2)*tq_mat(1,1)- tq_mat(2,1)*tq_mat(1,2)
  tinv(2,1)=tinv(1,2)
  tinv(3,1)=tinv(1,3)
  tinv(3,2)=tinv(2,3)
  
  tinv=tinv/det

  write(6,2103)
2103 format(" ---> Inverse  analytic matrix (Procacci)")
  do i=1,3
     write(6,1002) (tinv(i,j),j=1,3)
  end do
  do i=1,3
     do j=1,3
        xx=0.d0
        do k=1,3
           xx=xx+tq_mat(i,k)*tinv(k,j)
        end do
        write(6,*) i,j,xx
     end do
  end do

  tinv=0


  call inverse(tq_mat,tinv,3)

  ! compute t scalar

  t=0.d0
  do i=1,3
     tt(i)=0.d0
     do j=1,3
        tt(i)= tt(i)+tqvec(j)*tinv(j,i)
     end do
     t = t + tt(i)**2
  end do
  write(6,2003)
2003 format(" ---> Inverse  matrix (Procacci)")
  write(6,1002) tinv

  cthetaq=dsqrt(1.d0/(1+t)) ! this is thetaq=theta/2
  sthetaq = dsqrt(1-cthetaq*cthetaq)
  pi=dacos(-1.d0)
  write(6,*) "cos-sinQ",cthetaq,sthetaq,dacos(cthetaq)*180.0/pi,dasin(sthetaq)*180.0/pi

  ! compute ut 
  do i=1,3
     utq(i)=0.d0
     do j=1,3
        utq(i)=utq(i)+tinv(i,j)*tqvec(j)
     end do
  end do

  utq=utq*cthetaq

  write(6,*) "nomrQ", utq,utq(1)**2+utq(2)**2+utq(3)**2. + cthetaq**2.
  write(6,*) "Qudottc", utq(1)*tqvec(1)+utq(2)*tqvec(2)+utq(3)*tqvec(3)

  rio(1,1) = 1 - 2.0*utq(2)**2 - 2.0*utq(3)**2. 
  rio(2,2) = 1 - 2.0*utq(1)**2 - 2.0*utq(3)**2. 
  rio(3,3) = 1 - 2.0*utq(1)**2 - 2.0*utq(2)**2. 
  rio(1,2) = 2.*(utq(1)*utq(2)-utq(3)*cthetaq)
  rio(2,1) = 2.*(utq(1)*utq(2)+utq(3)*cthetaq)
  rio(1,3) = 2.*(utq(1)*utq(3)+utq(2)*cthetaq)
  rio(3,1) = 2.*(utq(1)*utq(3)-utq(2)*cthetaq)
  rio(2,3) = 2.*(utq(2)*utq(3)-utq(1)*cthetaq)
  rio(3,2) = 2.*(utq(2)*utq(3)+utq(1)*cthetaq)

  write(6,1023)
1023 format(" ---> RotationQ matrix (Procacci)")
  write(6,1002) rio

  ut=utq/sthetaq
  ctheta=(cthetaq**2-sthetaq**2)
  ctheta1=1-ctheta
  stheta= 2*cthetaq*sthetaq

  rio(1,1)=ut(1)*ut(1)*ctheta1 +ctheta
  rio(2,2)=ut(2)*ut(2)*ctheta1 +ctheta
  rio(3,3)=ut(3)*ut(3)*ctheta1 +ctheta
  rio(1,2)=ut(1)*ut(2)*ctheta1 -ut(3)*stheta
  rio(2,1)=ut(1)*ut(2)*ctheta1 +ut(3)*stheta
  rio(1,3)=ut(1)*ut(3)*ctheta1 +ut(2)*stheta
  rio(3,1)=ut(1)*ut(3)*ctheta1 -ut(2)*stheta
  rio(2,3)=ut(2)*ut(3)*ctheta1 -ut(1)*stheta
  rio(3,2)=ut(2)*ut(3)*ctheta1 +ut(1)*stheta
  

  write(6,1003)
1003 format(" ---> Rotation matrix (Procacci)")
  write(6,1002) rio

1002 format(10x,3f10.5)

! From Rot matrix finds new coordinated obeying rigidity constraint

  do i=1,3
     do l = 1,nato
        rtot = 0.d0
        do j =1,3
           rtot = rtot + rio(i,j)*r0(j,l)
        end do
        r(i,l) = rtot 
     end do
  end do
  
end subroutine rigidq



subroutine inverse(a,c,n)
!============================================================
! Inverse matrix
! Method: Based on Doolittle LU factorization for Ax=b
! Alex G. December 2009
!-----------------------------------------------------------
! input ...
! a(n,n) - array of coefficients for matrix A
! n      - dimension
! output ...
! c(n,n) - inverse matrix of A
! comments ...
! the original matrix a(n,n) will be destroyed 
! during the calculation
!===========================================================
implicit none 
integer n
double precision a(n,n), c(n,n)
double precision L(n,n), U(n,n), b(n), d(n), x(n)
double precision coeff
integer i, j, k

! step 0: initialization for matrices L and U and b
! Fortran 90/95 aloows such operations on matrices
L=0.0
U=0.0
b=0.0

! step 1: forward elimination
do k=1, n-1
   do i=k+1,n
      coeff=a(i,k)/a(k,k)
      L(i,k) = coeff
      do j=k+1,n
         a(i,j) = a(i,j)-coeff*a(k,j)
      end do
   end do
end do

! Step 2: prepare L and U matrices 
! L matrix is a matrix of the elimination coefficient
! + the diagonal elements are 1.0
do i=1,n
  L(i,i) = 1.0
end do
! U matrix is the upper triangular part of A
do j=1,n
  do i=1,j
    U(i,j) = a(i,j)
  end do
end do

! Step 3: compute columns of the inverse matrix C
do k=1,n
  b(k)=1.0
  d(1) = b(1)
! Step 3a: Solve Ld=b using the forward substitution
  do i=2,n
    d(i)=b(i)
    do j=1,i-1
      d(i) = d(i) - L(i,j)*d(j)
    end do
  end do
! Step 3b: Solve Ux=d using the back substitution
  x(n)=d(n)/U(n,n)
  do i = n-1,1,-1
    x(i) = d(i)
    do j=n,i+1,-1
      x(i)=x(i)-U(i,j)*x(j)
    end do
    x(i) = x(i)/u(i,i)
  end do
! Step 3c: fill the solutions x(n) into column k of C
  do i=1,n
    c(i,k) = x(i)
  end do
  b(k)=0.0
end do
end subroutine inverse

  SUBROUTINE up_low(string,ndim)
!--------------------------------------------------------------------------------------------
!   return string in lower case
!--------------------------------------------------------------------------------------------
! args
    INTEGER ndim
    CHARACTER*1 string(ndim)
!local
    INTEGER n,nstr
!  --executable statement
    DO n=1,ndim
       nstr=ICHAR(string(n))
       IF(nstr .GE. 65 .AND. nstr .LE.90) THEN
          string(n)=CHAR(nstr+32)
       END IF
    END DO

    RETURN
  END SUBROUTINE up_low

subroutine StripSpaces(string)
  character(len=*) :: string
  integer :: stringLen 
  integer :: last, actual
  
  stringLen = len (string)
  last = 1
  actual = 1
  
  do while (actual < stringLen)
     if (string(last:last) == ' ') then
        actual = actual + 1
        string(last:last) = string(actual:actual)
        string(actual:actual) = ' '
     else
        last = last + 1
        if (actual < last) &
             actual = last
     endif
  end do
  
end subroutine StripSpaces

