!!!! program metafes 
! compute a free energy surface in a n-dimensional space 
! from a metadynamics trajectory in a N-dimensional (N >= n) space of reaction coordinates
! synopsis:
! metafes [ntot] [a] ..[i].. [nrc]&< [FILE]' 
!         [ntot] is the total number of reaction coordinates' 
!         [i] is the index of the i-th RC on which the sampled data have to be projected
!         FILE contains a column of positions in a ntot-dimensional space
! TODO:
! pbc
! pass bin width for the plot as a parameter

module precision 
  
  ! Real kinds 
  
  integer, parameter :: kr4 = selected_real_kind(6,37)       ! single precision real 
  integer, parameter :: kr8 = selected_real_kind(15,307)     ! double precision real 
  integer, parameter :: kr16 = selected_real_kind(30,1000)   ! quadruple precision real 
  
  ! Integer kinds 
  
  integer, parameter :: ki4 = selected_int_kind(9)           ! single precision integer 
  integer, parameter :: ki8 = selected_int_kind(18)          ! double precision integer 
  
  !Complex kinds 
  
  integer, parameter :: kc4 = kr4                            ! single precision complex 
  integer, parameter :: kc8 = kr8                            ! double precision complex 
  
end module precision

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

module constants
  use precision
  implicit none
  save
  
  real(kr8), parameter :: pi = 3.14159265358979323846
  real(kr8), parameter :: gascon =  8.314472
  
end module constants

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

module meta
  use precision
  implicit none
  save 
  
  integer                :: n ! total number of hills
  integer                :: totrc,nrc ! number of RCs befor and after projection
  integer                :: sel(2) ! RCs for projection
  real(kr8)              :: h ! hills height
  real(kr8), allocatable :: ww(:) ! array of hills widths
  real(kr8), allocatable :: pos(:,:) ! array of hills positions
  
  
end module meta

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

program metafes
  use precision
  use meta
  use constants
  implicit none
  integer               :: i,j,jj,k
  integer               :: aerr,ierr
  integer               :: chan1, chan2
  integer               :: cmin(2), cmax(2)
  integer               :: angle
  integer               :: frac
  integer,allocatable   :: counter(:,:)
  real(kr8)             :: x1,x2,x,xx,d1,d2
  real(kr8)             :: pot
  real(kr8)             :: xmin(2), xmax(2), dx(2)
  real(kr8)             :: width(2) ! bin width
  real(kr8),allocatable :: rc(:)
  real(kr8),allocatable :: fes(:,:)
  character(1)          :: canc
  character(64)         :: arg,dummy

  select case (iargc())
  case(0)
     write(0,11)
     stop
  case(1) ! no projection; print full fes
     call getarg(1, arg)
     read(arg,*) totrc ! read the total number of reaction coordinates
     if(totrc > 2) then 
        write(0,*) 'projection max on 2D'
        stop
     else
        nrc = totrc
        sel(1) = 1
        sel(2) = 2
     endif
  case(2)
     call getarg(1, arg)
     read(arg,*) totrc ! read the total number of reaction coordinates
     nrc = 1
     call getarg(2, arg)
     read(arg,*) sel(1) ! read the index of the RC for projection
     sel(2) = 0
  case(3)
     call getarg(1, arg)
     read(arg,*) totrc ! read the total number of reaction coordinates
     nrc = 2
     call getarg(2, arg)
     read(arg,*) sel(1) ! read the index of the RC for projection
     call getarg(3, arg)
     read(arg,*) sel(2) ! read the index of the RC for projection
  case default ! > 3
     write(0,*) 'projection max on 2D'
     stop
  end select

  angle = 1 ! periodicity: 0 for distances [-infty,+infty], 1 for angles [-180,180]
  frac = 10 ! for the plot: the bin width is a fraction of the width of the hills width ("frac" times smaller)
  if(nrc < totrc) frac = 2 ! for projections bin size must be greater due to exponential average
     
  write(0,*) 'total number of RC', totrc
  write(0,*) 'number of RC for projection', nrc
  write(0,*) 'selected RCs for projection', (sel(j),j=1,nrc)
  
! allocate memory
  allocate(ww(totrc), stat = aerr)
  if(aerr > 0) then 
     write(0,12)
     stop
  endif
  
  xmin = 1.d10
  xmax = -1.d10
  
  k = 1 ! k is a row counter
  ! read the first input line 
  read(*,*) canc, n, h, ww
  
  write(0,*) 'total number of hills', n
  write(0,*) 'hills height', h, ' kJ mol-1'
  write(0,*) 'hills widths', (ww(j),j=1,nrc)
  
  ! use the hills widths as bin widths
  ! use a fraction
  do j = 1,nrc
     jj = sel(j)
     width(j) = ww(jj) / real(frac)
  enddo

  ! allocate memory
  allocate(pos(n,totrc),rc(totrc), stat = aerr)
  if(aerr > 0) then 
     write(0,12)
     stop
  endif
  
  ! read hills positions
  read: do i = 1,n
     k = k + 1
     read(*,*,iostat = ierr) rc
     if(ierr /= 0) exit read
     do j = 1,nrc
        jj = sel(j)
        xx = rc(jj)
        if(xx < xmin(j)) xmin(j) = xx
        if(xx > xmax(j)) xmax(j) = xx
     enddo
     pos(i,:) = rc
  enddo read
  
  if(ierr > 0) then 
     write(0,14) k
  elseif(ierr < 0) then 
     write(0,15) k-1,n
     n = k-1
  endif
  
  ! first and last bin
  do j = 1,nrc
     jj = sel(j)
     cmin(j) = nint(xmin(j)/width(j)) 
     cmax(j) = nint(xmax(j)/width(j))
  enddo
  
  if(nrc == 1) then 
     cmin(2) = 1
     cmax(2) = 1
  endif
  
  allocate(fes(cmin(1):cmax(1),cmin(2):cmax(2)), counter(cmin(1):cmax(1),cmin(2):cmax(2)), stat = aerr)
  if(aerr > 0) then 
     write(0,12)
     stop
  endif
  
  
  bin2: do chan2 = cmin(2),cmax(2)
     
     x2 = chan2 * width(2)
     bin1: do chan1 = cmin(1),cmax(1)

        x1 = chan1 * width(1)
        fes(chan1,chan2) = 0.0_kr8
        counter(chan1,chan2) = 0
        
        if(totrc.gt.nrc) then 
           ! for projecting a fes on a subset of the reaction coordinates one needs an exponential average
           hills: do i = 1,n
              rc = pos(i,:)
              d1 = rc(sel(1)) - x1
              d2 = rc(sel(2)) - x2
              if(angle == 1) then 
                 if(d1 > 180.0_kr8) d1 = d1 - 360.0_kr8
                 if(d1 < -180.0_kr8) d1 = d1 + 360.0_kr8      
                 if(d2 > 180.0_kr8) d2 = d2 - 360.0_kr8
                 if(d2 < -180.0_kr8) d2 = d2 + 360.0_kr8      
              endif
              if(abs(d1) < width(1) * 0.5_kr8) then
                 if((abs(d2) < width(2) * 0.5_kr8).or.(nrc == 1)) then
                    counter(chan1,chan2) = counter(chan1,chan2) + 1
                    fes(chan1,chan2) = fes(chan1,chan2) + exp(pot(rc,angle)/(gascon * 300.0_kr8))
                 endif
              endif
           enddo hills
           ! reweigth by dividing for the total number of hills in the bin (chan1,chan2) 
           ! it can be seen as an average
           fes(chan1,chan2) = -gascon * 300.0_kr8 * log(fes(chan1,chan2) / real(counter(chan1,chan2)))
           if(counter(chan1,chan2) == 0) fes(chan1,chan2) = 0.0_kr8
        elseif(totrc.eq.nrc) then 
           ! no projection: the fes is the potential inverted in sign
           rc(1) = x1
           rc(2) = x2
           fes(chan1,chan2) = - pot(rc,angle)
        endif
        
     enddo bin1
     
  enddo bin2
  
  if(nrc == 1) then
     do chan1 = cmin(1),cmax(1)
        x = chan1 * width(1)
        write(*,*) x,fes(chan1,1)
     enddo
  else
     do chan1 = cmin(1),cmax(1)
        do chan2 = cmin(2),cmax(2)
           write(*,*) chan1*width(1),chan2*width(2),fes(chan1,chan2)
        enddo
        write(*,*) 
     enddo
  endif
  
  ! deallocate memory 
  deallocate(ww,pos,rc,fes,counter, stat=aerr)
  if(aerr > 0) then 
     write(0,13)
     stop
  endif
  
  
11 format(1x,'syntax: metafes [ntot] [a] ..[i].. [nrc]&< [FILE]',/ &  
        '[ntot] is the total number of reaction coordinates',/&
        '[i] is the index of the i-th RC on which the sampled data have to be projected',/&
        'FILE contains a column of positions in a ntot-dimensional space')
12 format(1x,'error while allocating memory')
13 format(1x,'error while deallocating memory')
14 FORMAT(1x,'an error occurred reading line', i8)
15 FORMAT(1x,'end of file reached at line', i8, ' while expecting',i8, ' lines')
  
end program metafes

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

real*8 function pot(rc,angle)
  use precision 
  use meta
  implicit none
  
  integer :: angle
  real(kr8) :: rc(totrc)
! local variables
  integer :: i,irc,irc2,t,zer0
  real(kr8) :: rc0, dr0, dra, lucy
  real(kr8) :: dr(totrc), wi(totrc)
  
  pot = 0.0_kr8
  
  wi = 1.0_kr8/ww
  
  if(n > 0) then
     zer0 = 0
     
     hills_loop: do t = 1,n      
        
        if(totrc > 1) then
           if(zer0 > 0) then  ! if the last hill was far in cv "zer0", check if it is still far in that cv
              rc0 = pos(t,zer0)
              dr0 = rc(zer0) - rc0
              if(angle == 1)then 
                 if(dr0 > 180.0_kr8) dr0 = dr0 - 360.0_kr8
                 if(dr0 < -180.0_kr8) dr0 = dr0 + 360.0_kr8
              endif
              dra = abs(dr0)
              if(dra > ww(zer0)) cycle hills_loop
           endif
        endif
        
        do irc = 1,totrc     
           rc0 = pos(t,irc)
           dr(irc) = rc(irc) - rc0
           dr0 = dr(irc)
           if(angle == 1)then 
              if(dr0 > 180.0_kr8) dr0 = dr0 - 360.0_kr8
              if(dr0 < -180.0_kr8) dr0 = dr0 + 360.0_kr8
           endif
           dr(irc) = dr0
           dra = abs(dr0)
           if(dra < ww(irc)) then
              zer0 = 0
           else                ! do not calculate forces, goto the next hill
              zer0 = irc
              cycle hills_loop
           endif
        enddo
        
        lucy = 1.0_kr8
        do irc = 1,totrc     
           dra = abs(dr(irc))
           lucy = lucy * (1.0_kr8 + 2.0_kr8 * wi(irc) * dra) * (1.0_kr8 - wi(irc) *dra)**2  
        enddo
        
        pot = pot + lucy
        
     enddo hills_loop
     
     pot = h * pot
     
  endif

end function pot



