module precision 

  integer, parameter :: kr4 = selected_real_kind(6,37)      
  integer, parameter :: kr8 = selected_real_kind(15,307)  
  integer, parameter :: kr16 = selected_real_kind(30,1000) 
  integer, parameter :: ki4 = selected_int_kind(9)           
  integer, parameter :: ki8 = selected_int_kind(18)          
  integer, parameter :: kc4 = kr4                            
  integer, parameter :: kc8 = kr8                            
  ! generic kinds
  integer, parameter :: ki=ki4,kr=10,kc=kc8
  
end module precision

program hist2d
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!! compute 2d histograms and print it in gnuplot or in plotmtv data format       !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!! syntax: hist2d <input filename> <output filename> [bin width] [bin width]     !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!! input file must contain data arranged in two or three columns                 !!
!! if the third column is present, a weighted average will be performed          !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!! the extension of the output filename will determine the output format:        !!
!!   *.plt will produce data in gnuplot format to be used with splot command     !!
!!   *.mtv will produce an input file to be used with the plotmtv program        !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!! bin width(s) are passed as optional arguments:                                !!
!!   if no widths are passed the program will try to find reasonable bin widths  !!
!!   if only a width is passed, the program will use is for both the coordinates !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  use precision
  implicit none
  integer(ki) :: i,j
  integer(ki) :: err
  integer(ki) :: in=11,out=12              ! unit numbers
  integer(ki) :: n                         ! number of data points
  integer(ki) :: nchan(2)                  ! bins number 
  real(kr)    :: dx(2)                     ! bins width
  real(kr)    :: x(2),w,wsum
  real(kr)    :: xmin(2),xmax(2)           ! min and max values in the data
  integer(ki)    :: nmax                      ! max number of data point in a bin
  integer(ki) :: chan(2)
  integer(ki) :: ncol                      ! number of columns in input file
  character   :: arg*64,input*64,output*64
  integer(ki) :: out_len
  real(kr)    :: x1,x2
  real(kr) :: ffm
  logical     :: findbin
  logical     :: weighted
  ! allocatable arrays
  real(ki), allocatable     :: hh(:,:)
  real(kr), allocatable     :: ff(:,:)

  findbin=.false.
  select case(iargc())
  case(4)
     call getarg(1,arg)
     read(arg,*) input
     call getarg(2,arg)
     read(arg,*) output
     call getarg(3,arg)
     read(arg,*) dx(1)
     call getarg(4,arg)
     read(arg,*) dx(2)
  case(3)
     call getarg(1,arg)
     read(arg,*) input
     call getarg(2,arg)
     read(arg,*) output
     call getarg(3,arg)
     read(arg,*) dx(1)
     dx(2) = dx(1)
  case(2)
     call getarg(1,arg)
     read(arg,*) input
     call getarg(2,arg)
     read(arg,*) output
     findbin=.true.
  case default
     write(0,'(16(a,/))') &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", &
          "CC compute 2d histograms and print it in gnuplot or in plotmtv data format       CC", &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", &
          "CC syntax: hist2d <input filename> <output filename> [bin width] [bin width]     CC", &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", &
          "CC input file must contain data arranged in two or three columns                 CC", &
          "CC if the third column is present, a weighted average will be performed          CC", &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", &
          "CC the extension of the output filename will determine the output format:        CC", &
          "CC   .plt will produce data in gnuplot format to be used with splot command      CC", &
          "CC   .mtv will produce an input file to be used with the plotmtv program         CC", &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", &
          "CC bin width(s) are optional arguments:                                          CC", &
          "CC   if no widths are passed the program will try to find reasonable bin widths  CC", &
          "CC   if only a width is passed, the program will use is for both the coordinates CC", &
          "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC"
     stop
  end select

  write(0,*) 'input file: ',  input
  write(0,*) 'output file: ',  output
  if(.not.findbin) then 
     write(0,*) 'bin width along x: ',  dx(1)
     write(0,*) 'bin width along y: ',  dx(2)
  end if

!!!!!! open units

  open(unit=in,file=input)
  open(unit=out,file=output)

  !! check number of columns

  read(in,'(a)') arg
  do i =1,40 ! The very maximum that the string can contain
     read( arg, *, iostat=err ) ( x1, j=1,i )
     if ( err /= 0 ) then
        ncol = i - 1
        write(0,*) 'input file contains: ', ncol, 'columns'
        select case(ncol)
        case(2)
           weighted = .false.
        case(3)
           weighted = .true.
           write(0,*) 'a weighted average will be performed'
           write(0,*) 'using numbers from the third column as weights'
        case default
           write(0,*) 'it should contain 2 or 3 columns'
           write(0,*) 'program will stop'
           stop
        end select
        exit
     endif
  enddo
  rewind(in)

!!!!!! find data points number and max/min values

  n = 0
  xmin =  10000000.0
  xmax = -10000000.0
  if(weighted) wsum = 0.0

  pre_loop: do 
     if(weighted) then 
        read(in,*,iostat = err) x,w
        wsum = wsum + w
     else
        read(in,*,iostat = err) x
     end if
     if(err /= 0) then 
        if(err < 0) then 
           exit pre_loop
        else
           write(0,*) 'error while reading line ', n
           write(0,*) 'the program will stop'
           stop
        endif
     endif
     do j = 1,2
        if(x(j) < xmin(j)) xmin(j) = x(j)
        if(x(j) > xmax(j)) xmax(j) = x(j)
     enddo
     n = n + 1
  enddo pre_loop

!  xmin = -180.0
!  xmax = 180.0

  rewind(in)

  ! if bin widths are not passed, 
  ! put the max number of data points per bin 
  ! equal to 1 /20 of the total number of points

  if(findbin) then 
     if(weighted) then 
        nmax = nint( wsum / 20.0)
     else
        nmax = n / 20
     end if
  end if

  write(0,*) n, ' points'
  write(0,*) 'max', xmax
  write(0,*) 'min', xmin


  if(.not.findbin) then 
     ! compute bins number
     nchan = int((xmax-xmin)/dx) + 1

     write(0,*) nchan, ' bins'
     write(0,*) 'bin width: ', dx

     allocate(hh(nchan(1),nchan(2)),ff(nchan(1),nchan(2)),stat=err)

     hh = 0

     ! go for histogram
     do j = 1,n
        if(weighted) then 
           read(in,*) x,w
           chan = nint((x - xmin) / dx + 0.5)
           hh(chan(1),chan(2)) = hh(chan(1),chan(2)) + w
        else
           read(in,*) x
           chan = nint((x - xmin) / dx + 0.5)
           hh(chan(1),chan(2)) = hh(chan(1),chan(2)) + 1.0
        end if
     enddo

  else
     ! find reasonable bin width
     write(0,*) 'find reasonable bin width'
     dx = xmax - xmin
     histogram_loop: do 
        nchan = int((xmax-xmin)/dx) + 1
        allocate(hh(nchan(1),nchan(2)),ff(nchan(1),nchan(2)),stat=err)
        hh = 0.0
        do j = 1,n
           if(weighted) then 
              read(in,*) x,w
              chan = nint((x - xmin) / dx + 0.5)
              hh(chan(1),chan(2)) = hh(chan(1),chan(2)) + w
           else
              read(in,*) x
              chan = nint((x - xmin) / dx + 0.5)
              hh(chan(1),chan(2)) = hh(chan(1),chan(2)) + 1.0
           end if
           ! if too many points are in a single bin, then 
           ! change half
           if (hh(chan(1),chan(2)) > nmax) then 
              rewind(in)
              dx = 0.5 * dx
              deallocate(hh,ff)
              cycle histogram_loop
           end if
        enddo
        exit histogram_loop
     end do histogram_loop

     write(0,*) 'final bin width: ', dx

  end if

!!!!!! convert to free energy

  ff = hh

  ffm = -8.314*0.3*log(maxval(hh)) ! free energy minimum

  where (hh /= 0.0) ff = -8.314*0.3*log(hh) 

  where (hh /= 0.0) ff = ff - ffm
  
  ffm = maxval(ff,hh /= 0.0)

  where (hh == 0.0) ff = ffm

!!!!!!!!!!!!!!!!!!!!!! print data for plotting

  out_len = len_trim(output)

  select case(output(out_len-3:out_len))
  case('.mtv')

     write(out,*) '$ DATA=CONTOUR name=" "'
     write(out,11) xmin(1),xmax(1),nchan(1)
     write(out,12) xmin(2),xmax(2),nchan(2)
     write(out,*) '%contstyle=2'
     if(nint(maxval(ff)-minval(ff)) > 50) then 
        write(out,*) '%nsteps =50'
     else
        write(out,*) '%nsteps =', nint(maxval(ff)-minval(ff))
     end if
     
     write(out,*) '%cmin = ', minval(ff)
     write(out,*) '%cmax = ', maxval(ff)
     write(out,*) '%vxmin=', xmin(1)
     write(out,*) '%vxmax=', xmax(1)
     write(out,*) '%vymin=', xmin(2)
     write(out,*) '%vymax=', xmax(2)
     write(out,*) '% interp     = 2'

     do j=1,nchan(2)
        write(out,*) (ff(i,j),i=1,nchan(1))
     enddo

11   format('% XMIN =',f12.5,' XMAX =',f12.5,' NX = ',i4)
12   format('% YMIN =',f12.5,' YMAX =',f12.5,' NY = ',i4)

  case('.plt')
     do i = 1,nchan(1)
        x1 = xmin(1) + (i-0.5) * dx(1) 
        do j = 1,nchan(2)
           x2 = xmin(2) + (j-0.5) * dx(2) 
           write(out,'(3f12.5)') x1,x2,ff(i,j)
        end do
        write(out,*)
     end do
  case default
     write(0,*) 'output file extension should be .mtv or .plt'
     stop
  end select

end program hist2d



