#!/bin/bash

# Syntax: run_serial_tests.bash [options] <ORAC> 
#  where <ORAC> is the full name of the program 
#   options: -z -> clear temp files and exit
#            -h -> display this help

# ORAC program: Do all SMD tests sequentially 
# script has been tested on Linux platforms only. 

# ----------------------------------------------------------------------
# Help: copy first comments block in this file
# ----------------------------------------------------------------------
function Help {
name=`basename "$0"`;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# other functions
# ----------------------------------------------------------------------
function CleanUpFiles {
    rm -rf $* >& /dev/null
}

# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------
TEMP_FILES="\
  bin/bennett \
  check* \
  dwr* \
  DWR* \
  energie \
  FORWARD/* \
  fuffa* \
  OUT_TEST \
  *.out \
  *.pdb \
  *.prmtpg \
  REVERSE/* \
  */*.rst \
  *.rst \
  *.rst* \
  tmp* \
  *.works* \
  wr* \
  WR* \
"



# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "hz" Option
do
  case $Option in
    h     )    Help;;    # help: copy first comments block
    z     )    CleanUpFiles $TEMP_FILES; exit ;;  # just cleanup
    *     )    Help;;
  esac
done

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
if [ $# -lt 1 ] ; then
    Help;
    exit
else 
    ORAC=$1
fi


# ----------------------------------------------------------------------
# initial checks and cleanup
# ----------------------------------------------------------------------

#
# make sure all needed programs are executable and on the $PATH
#
###BINARIES="mpiexec mpdallexit mpdboot $ORAC"
BINARIES="$ORAC"

for prog in $BINARIES; do
    if ! type $prog >/dev/null 2>&1; then
	echo "ERROR: \"$prog\" not found."
	echo "       This is needed by $0 to work. Check your"
	echo "       \$PATH variable or install program \"$prog\"."
	exit 2
    fi
done


#
# files cleanup
#
CleanUpFiles $TEMP_FILES;

# ----------------------------------------------------------------------
#   TESTS
# ----------------------------------------------------------------------

echo "Doing first test for SMD ....." 
echo " PLATFORM = " `uname -a`   > OUT_TEST
echo " ORAC executable = " $ORAC   >> OUT_TEST
echo " RELEASE = " `cat ../../RELEASE` >> OUT_TEST
  # First test 
  # Test on driven stretching, bending and torsion of decaalanine
echo "Testing 1a.in..."
echo "Testing 1a.in..." >> OUT_TEST
$ORAC < 1a.in >& tmp ; 
 grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -2 WRK.ala10 >> OUT_TEST

echo "Testing 1b.in..."
echo "Testing 1b.in..." >> OUT_TEST
$ORAC < 1b.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -2 WRK.ala10 >> OUT_TEST

echo "Testing 1c.in..."
echo "Testing 1c.in..." >> OUT_TEST
$ORAC < 1c.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -2 WRK.ala10 >> OUT_TEST
  
echo "Testing 1d.in..." 
echo "Testing 1d.in..." >> OUT_TEST
$ORAC < 1d.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -4 WRK.ala10 >> OUT_TEST

echo "Testing thermal_changes_GCT.in..."
echo "Testing thermal_changes_GCT.in..." >> OUT_TEST
$ORAC < thermal_changes_GCT.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -4 WRK.ala10 >> OUT_TEST; cat WRKTEMP.ala10>> OUT_TEST

echo "Testing steer_path.in..."
echo "Testing steer_path.in..." >> OUT_TEST
$ORAC < steer_path.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; tail -4 DWRK.ala10 >> OUT_TEST

  # Second  test 
  # generate state A and B starting equilibrium distributions
  # state A phase space points (folded 10ala) in RESTART_A directory
  # state B phase space points (unfolded 10ala) in RESTART_B directory
echo "Doing second  test for SMD ....." 
echo "Testing 2a.in..."
echo "Testing 2a.in..." >> OUT_TEST
#  bin/do_restart_a.bash > tmp1a  & $ORAC < 2a.in >& tmpa ;  
$ORAC < 2a.in >& tmpa 
grep "Total    = " tmpa >> OUT_TEST ;

echo "Testing 2b.in..." 
echo "Testing 2b.in..." >> OUT_TEST ;
#  bin/do_restart_b.bash > tmp1b & $ORAC < 2b.in >& tmpb   ;
$ORAC < 2b.in >& tmpb   ;
grep "Total    = " tmpb >> OUT_TEST

  # Third test 
  #  

echo "Doing third and final test for SMD ....." 
# generates forward works (see ancillary script forward_trajectories.bash)
echo "Doing forward nonequilibrium experiments"  >> OUT_TEST
echo "  Doing forward nonequilibrium experiments" 
ls RESTART_A/*.rst | wc | awk '{print $1}'   > works_forward
./forward_trajectories.bash  $ORAC >> OUT_TEST
for i in `ls FORWARD/*.wrk` ; do tail -1 $i | awk '{print $7}' >> works_forward ; done

# generates reverse works (see ancillary script reverse_trajectories.bash)
echo "Doing reverse  nonequilibrium experiments"  >> OUT_TEST
echo "  Doing reverse  nonequilibrium experiments" 
ls RESTART_B/*.rst | wc | awk '{print $1}'   > works_reverse
./reverse_trajectories.bash  $ORAC  >> OUT_TEST
for i in `ls REVERSE/*.wrk` ; do tail -1 $i | awk '{print $7}' >> works_reverse ; done

# Launch Bennett program (which reads works_forward and works_reverse)
# and printout Free energy
./bennett |  tee -a OUT_TEST

# compile and launch Bennett
##make free_energy

# Now check if tests are OK (or almost OK) 
diff OUT_TEST OUT | grep -v PLATFORM >& tmp.diff;

echo "** TEST COMPLETED **"

if `test -s tmp.diff`; 
then 
    echo "-------------------------------------------------------------------"
    echo " Warning - your output differs from reference:" 
    echo " compare OUT_TEST to reference OUT (try \`make diff'), "
    echo " or you may want to run individual tests with e.g. \`make 2b.out'"
    echo ""
    echo " Please read the NOTE on tests in the README file"
    echo "-------------------------------------------------------------------"
    exit  
fi

exit
