#!/bin/bash

# Syntax: run_parallel_tests.bash [options] <ORAC_P> <n_procs>
#  where <ORAC_P> is the full name of the program 
#  and <n_procs> is the n. of replicas
#   options: -z -> clear temp files and exit
#            -h -> display this help

# ORAC program: do SMD tests in parallel.
# This script has been tested on Linux/mpich2 platforms only. 


# ----------------------------------------------------------------------       
# Help: copy first comments block in this file
# ----------------------------------------------------------------------
function Help {
name=`basename "$0"`;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# other functions
# ----------------------------------------------------------------------
function CleanUpFiles {
    rm -rf $* >& /dev/null
}

# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------

# change 'MPIRUN=' line below according to your MPI package 
# (a) MPICH2, conforms to MPI-2 recommendation
# mpich2 v.1.0.7 (run 'mpdboot -f mpd.hosts' first; also set up ~/.mpd.conf)
MPIRUN="mpiexec -n"
# mpich2 v.1.4.1
#MPIRUN=mpiexec -f machinefile -n
#   use -disable-hostname-propagation  if you encounter communication errors
#MPIRUN=mpiexec -f machinefile -disable-hostname-propagation -n

# OpenMPI
# MPIRUN="mpirun -nd" 

TEMP_FILES="          \
./ala10_A.prmtpg      \
./ala10_B.prmtpg      \
./backward.dat        \
./bin/fes             \
./forward.dat         \
./FORWARD/*           \
./INPUT               \
./OUT_PARALLEL_TEST   \
./PAR????             \
./RESTART_A/*         \
./RESTART_B/*         \
./REVERSE/*           \
./tmp*                \
./works_*             \
"


# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "hz" Option
do
  case $Option in
    h     )    Help;;    # help: copy first comments block
    z     )    CleanUpFiles $TEMP_FILES; exit ;;  # just cleanup
    *     )    Help;;
  esac
done

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
if [ $# -lt 2 ] ; then
    help;
    exit
else 
    ORAC_P=$1
    NPROCS=$2
fi


# ----------------------------------------------------------------------
# initial checks and cleanup
# ----------------------------------------------------------------------

#
# make sure all needed programs are executable and on the $PATH
#
###BINARIES="mpiexec mpdallexit mpdboot $ORAC"
BINARIES="mpiexec $ORAC_P"

for prog in $BINARIES; do
    if ! type $prog >/dev/null 2>&1; then
	echo "ERROR: \"$prog\" not found."
	echo "       This is needed by $0 to work. Check your"
	echo "       \$PATH variable or install program \"$prog\"."
	exit 2
    fi
done

#
# files cleanup
#
CleanUpFiles $TEMP_FILES;

#
# cleanup any MPI process left, and initialize MPI  (MPICH2 only)
#

# killall -9 $ORAC_P >&/dev/null
##        kill mpd demon 
# mpdallexit 
##        start the mpd demon on hosts 
#echo "starting the mpd demon" 
# mpdboot -n  `cat mpd.hosts | wc -l` && ( echo "Starting the program on these hosts:"; cat mpd.hosts);

# ----------------------------------------------------------------------
#   TESTS
# ----------------------------------------------------------------------

#        First test 
#        Test on driven stretching, bending and torsion of decaalanine
TARGET="0001"
echo "Doing first test for SMD ....." 
rm -fr PAR* >& /dev/null
${MPIRUN} 1 $ORAC_P < 1Pa.in >& tmp ;
echo "Testing 1a.in..." >  OUT_PARALLEL_TEST
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST; rm PAR$TARGET/out$TARGET; tail -2 PAR$TARGET/WRK.ala10 >> OUT_PARALLEL_TEST
rm -fr PAR* >& /dev/null
echo "Testing 1b.in..." >> OUT_PARALLEL_TEST
rm -fr PAR* >& /dev/null
${MPIRUN} 1 $ORAC_P < 1Pb.in >& tmp ;
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST; rm PAR$TARGET/out$TARGET; tail -2 PAR$TARGET/WRK.ala10 >> OUT_PARALLEL_TEST
rm -fr PAR* >& /dev/null
echo "Testing 1c.in..." >> OUT_PARALLEL_TEST
${MPIRUN} 1 $ORAC_P < 1Pc.in >& tmp ;
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST; rm PAR$TARGET/out$TARGET; tail -2 PAR$TARGET/WRK.ala10 >> OUT_PARALLEL_TEST
rm -fr PAR* >& /dev/null
echo "Testing 1d.in..." >> OUT_PARALLEL_TEST
${MPIRUN} 1 $ORAC_P < 1Pd.in >& tmp ;
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST; rm PAR$TARGET/out$TARGET; tail -4 PAR$TARGET/WRK.ala10 >> OUT_PARALLEL_TEST

# Second  test 
# generate state A and B starting equilibrium distributions
# state A phase space points (folded 10ala) in RESTART_A directory
# state B phase space points (unfolded 10ala) in RESTART_B directory
echo "Doing second  test for SMD ....." 
echo "Testing 2Pa.in..." >> OUT_PARALLEL_TEST
rm RESTART_A/* >& /dev/null
rm RESTART_B/* >& /dev/null
rm -fr PAR* >&/dev/null
${MPIRUN} 1 $ORAC_P < 2Pa.in >& tmpa ;  
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST ;
mv PAR$TARGET/ala10_A.prmtpg .
echo "Testing 2Pb.in..." >> OUT_PARALLEL_TEST ;
rm -fr PAR* >&/dev/null
${MPIRUN} 1 $ORAC_P < 2Pb.in >& tmpb   ;
grep "Total    = " PAR$TARGET/out$TARGET >> OUT_PARALLEL_TEST
mv PAR$TARGET/ala10_B.prmtpg .

# Third test 
#  
echo "Doing third and final test for SMD ....." 

printf 'cleaning... '
${MPIRUN} $NPROCS rm -fr PAR* >& /dev/null
printf 'done.\n'

# generate  $NPROCS forward work measurements in parallel
echo 'Starting job 3a on ' $NPROCS ' processors'
${MPIRUN} $NPROCS  $ORAC_P < 3a.in
for i in `ls -d PAR*`; do  mv $i/WRKa.1 FORWARD/WRKa.$i; done
printf 'cleaning... '
${MPIRUN} $NPROCS rm -fr PAR* >& /dev/null
printf 'done.\n'

# generate  $NPROCS reverse work measurements in parallel
echo 'Starting job 3b on ' $NPROCS ' processors'
${MPIRUN} $NPROCS  $ORAC_P < 3b.in
for i in `ls -d PAR*`; do  mv $i/WRKb.1 REVERSE/WRKb.$i; done
printf 'cleaning... '
${MPIRUN} $NPROCS rm -fr PAR* >& /dev/null
printf 'done.\n'

# collect work data in one file for forward and one file for backward trajectory
for i in `ls FORWARD/WRK*` ; do grep 'bond' $i | grep -v '#' | awk '{print  $5,$7}' >> forward.dat ; done
for i in `ls REVERSE/WRK*` ; do grep 'bond' $i | grep -v '#' | awk '{print  $5,$7}' >> backward.dat ; done

# recompile Bennett program 
##make -C bin clean >& /dev/null
##make -C bin fes   >& /dev/null


# Launch MA and CP PMF reconstruction from bidirectional data
### bin/fes >> OUT_PARALLEL_TEST
echo $NPROCS | ./fes >> OUT_PARALLEL_TEST

# ----------------------------------------------------------------------


# Now check if tests are OK (or almost OK) 
diff OUT_PARALLEL_TEST OUT_PARALLEL >& tmp.diff;

echo "** TEST COMPLETED **"

if `test -s tmp.diff`; 
then 
    echo "-------------------------------------------------------------------"
    echo " Warning - your output differs from reference:" 
    echo " compare OUT_PARALLEL_TEST to reference OUT_PARALLEL, "
    echo " or you may want to run individual tests with e.g. \`make 2b.out'"
    echo ""
    echo " Please read the NOTE on tests in the README_PARALLEL file"
    echo "-------------------------------------------------------------------"
    exit  
fi

exit
