#! /bin/bash

# reorder pdb and diagnostic files in PAR*/ by replica  
#
# syntax: order.sh [-d] <pdb-file-name> <diag-file-name>
#
#   options: 
#    -d   ignore water molecules in PDB files

# ----------------------------------------------------------------------
# help: copy first comments block in this file
# ----------------------------------------------------------------------
function help() {
echo $0
name=`basename "$0"`;
echo "-->" $name;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------

tmp_files="ORD-*-* tmp.rep rem.*"
BIN=`dirname "$0"`;

# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "dh-" Option
do
  case $Option in
    d     )    DRY=1;;  # don't process water molecules in PDB
    *     )    help;;   # help: copy first comments block
  esac
done

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
pdb_to_order=$1
diag_to_order=$2

# ----------------------------------------------------------------------
# read user's configuration
# ----------------------------------------------------------------------

# ----------------------------------------------------------------------
# final parameters setting
# ----------------------------------------------------------------------

if [ "$diag_to_order" = "" ]; then
    diag_to_order="REM_DIAGNOSTIC"
fi


# ----------------------------------------------------------------------
# engine
# ----------------------------------------------------------------------


rm $tmp_files &> /dev/null;


# split pdb into single frames, in the same dir 
#   use either /REPLI/ or /atoms/ as the split pattern
for i in PAR*/${pdb_to_order}; do csplit -f "$i." $i  '/[RaE][Etn][Pos][Lme][Ism]/' '{*}' >& /dev/null; done

# copy into master dir with a meaningful name ORD-xxxx-yyyyyyyyyy;
# x=replica; y=step
for i in PAR*/${pdb_to_order}.*; do 
    [ -f $i ] && mv $i ORD-`${BIN}/frame-rename.awk $i`;
done

# group diagnostic data one file per replica
awk '! /#/ {print $0 >> "rem." sprintf("%.4d",$2)}' PAR*/${diag_to_order}

# reorder into one file per replica
#   move '.pdb' trail in $pdb_to_order to the end of the new files
for r in `sort -u -n tmp.rep`; do 
    if [ $DRY ] ; then 
	grep -h -v -E "(SPC|TIP|HOH)" ORD-${r}-* > ${pdb_to_order%%.pdb}-${r}.pdb; 
    else cat ORD-${r}-* > ${pdb_to_order%%.pdb}-${r}.pdb; 
    fi
    echo "#   time(fs) rem index         Etot         Ekin         Epot        Ekinh        Epoth" > ${diag_to_order}-${r}
    sort -nk1 rem.${r} >> ${diag_to_order}-${r}
done

rm $tmp_files

