!--------------------------------------------------------------------------------------- 
! program to compute the ligand-receptor translational volume using REST stage sampling
! ligand-target COM vector are printed to standard output for each configuration. 
!--------------------------------------------------------------------------------------- 
program volume
!---------------------------------------------------------------------------------------
! The program syntax is :
! $ volume nrep filename [T/F] 
!      nrep=replica index 
!      filename+pdb file (stripped from water: e.g. 'grep -v TIP file.pdb'))
!      last arg is optional and print two xyz file with full coordinate in 
!      the protein inertia frame.  
!---------------------------------------------------------------------------------------
  implicit none

  character*20 dum,filename
  character*4 atom
  character*2 ires_lig
  character*1 rep,irep,debug
  real*8   co(3,3),iner(3),mass,time,t(3,3),trot(3,3),coml(3),comh(3),abserr,mltot,mhtot,dd(3),new(3)
  real*8, allocatable :: rh(:,:),rl(:,:),ml(:),mh(:)
  real*8   dd1(3),dd2(3),rot1(3),rot2(3),check1(3),check2(3),cc1,cc2,ddold(3),ddmax
  integer i,j,k,nrep,mrep,iret,ierr,n,nt,nl,ires,ires_old,kmax,ilig

  rep="1"
  debug="F"
  ires_lig="1"
  CALL GET_COMMAND_ARGUMENT(1, rep)
  CALL GET_COMMAND_ARGUMENT(2, filename)
  CALL GET_COMMAND_ARGUMENT(3, debug)
  CALL GET_COMMAND_ARGUMENT(4, ires_lig)

  abserr=0.001
  open(unit=10,file=trim(filename))
  open(unit=66,file="ine.xyz")
  open(unit=67,file="inerot.xyz")

  ires_old=0
  ddold=0.d0
  i=0
  read(ires_lig,'(i)') ilig  
  !   find natomos and nligand from first frame
  do while(dum(1:3)/="END") 
     read(10,'(A20,i6)') dum,ires
!     write(6,*) dum,ires
     i=i+1
     if(ires.eq.(ilig+1).and.ires.ne.ires_old) nl=i-2
     ires_old=ires
  end do
  n=i-2
  nt=n-nl

! now allocate coordinate and mass arrays

  allocate(rh(3,nt),rl(3,nl),mh(nt),ml(nl),stat=iret) 

! loop on PDB file (N.B. this must be stripped from water with e.g.: 'grep -v TIP file.pdb')

  rewind(10)
  do 
     read(10,'(18x,f12.2,12x,9f10.5,12x,2i5)',end=1) time,((co(i,j),j=1,3),i=1,3),mrep,nrep
     ! jump to end-of-conf if replica is not that selected with command arg
     write(irep,'(I1)') nrep
     if(irep.ne.rep) THEN
        do i=1,n+1
           read(10,*,END=1) 
        end do
     else
        ! read ligand coordinates
        do j=1,nl
           read(10,'(12x,A4,14x,3f8.3)') atom,(rl(k,j),k=1,3)
           call get_mass(atom,mass)
           ml(j)=mass
        end do
        ! read target coordinates
        do j=1,nt
           read(10,'(12x,A4,14x,3f8.3)') atom, (rh(k,j),k=1,3)
           call get_mass(atom,mass)
           mh(j)=mass
        end do
        read(10,*,END=1)  ! end of conf
        !shift origin on com of target
        comh=0.d0
        mhtot=0.d0
        do j=1,nt
           do k=1,3 
              comh(k)=comh(k)+mh(j)*rh(k,j)
           end do
           mhtot=mhtot+mh(j)
        end do
        comh=comh/mhtot
        do j=1,nl
           do k=1,3
              rl(k,j)=rl(k,j)-comh(k)
           end do
        end do
        do j=1,nt
           do k=1,3
              rh(k,j)=rh(k,j)-comh(k)
           end do
        end do

!       compute inertia tensor of the target protein
        call inertia(mh,rh,nt,t)


        call jacobi(t,trot,abserr,3)

        coml=0.d0
        !      compute com vector of ligand
        mltot=0.d0
        if(debug.EQ."T") THEN
           write(66,*) n
           write(66,*)
        END IF
        do j=1,nl
           call rotate(rl(1,j),trot)  ! report lig coordinate in the TARGET inertia frame 
           do k=1,3 
              coml(k)=coml(k)+ml(j)*rl(k,j)
           end do
           if(debug.EQ."T") THEN
              call get_type(ml(j),atom)
              write(66,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rl(k,j),k=1,3),(coml(k),k=1,3)
           ENDIF
           mltot=mltot+ml(j)
        end do
        coml=coml/mltot
!       write(6,'(A4,1x,3f12.5,f10.4)') 'Coml',coml,mltot
        do j=1,nt
           call rotate(rh(1,j),trot) ! report target coordinate in the TARGET inertia frame 
           call get_type(mh(j),atom)
           if(debug.EQ."T") THEN
              write(66,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rh(k,j),k=1,3),(coml(k),k=1,3)
           ENDIF
        end do
        dd=coml
        ddmax=0.d0
        kmax=0
        do k=1,3
           rot1(k)=1
           rot2(k)=1
        end do
        do k=1,3
           if(abs(dd(k)).gt.ddmax) THEN 
              ddmax=abs(dd(k))
              kmax=k
           endif
        end do
!       sign of the largest com component must be positive
        if(debug.EQ."T") THEN
           write(67,*) n
           write(67,*)
        ENDIF
        if(dd(kmax).lt.0.d0) THEN
           dd1(kmax)=-dd(kmax)
           dd2(kmax)=-dd(kmax)
           rot1(kmax)=-1.
           rot2(kmax)=-1.
!          of so find the best rotation around other two axis
           if(kmax.eq.1) THEN 
              dd1(2)=-dd(2)
              dd1(3)= dd(3)
              dd2(2)= dd(2)
              dd2(3)=-dd(3)
           END IF
           if(kmax.eq.2) THEN 
              dd1(1)=-dd(1)
              dd1(3)= dd(3)
              dd2(1)= dd(1)
              dd2(3)=-dd(3)
           END IF
           if(kmax.eq.3) THEN 
              dd1(1)=-dd(1)
              dd1(2)= dd(2)
              dd2(1)= dd(1)
              dd2(2)=-dd(2)
           END IF
           rot1=nint(dd1/dd)
           rot2=nint(dd2/dd)
           check1=dd1-ddold
           check2=dd2-ddold
           cc1=0.d0
           cc2=0.d0
           do k=1,3
              cc1=cc1+check1(k)**2
              cc2=cc2+check2(k)**2
           end do
           if((cc1-cc2).GT.0.D0) THEN 
              dd=dd2
              if(debug.eq."T") THEN
                 do i=1,nl 
                    do k=1,3
                       rl(k,i)=rot2(k)*rl(k,i)
                    end do
                    call get_type(ml(i),atom)
                    write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rl(k,i),k=1,3)
                 end do
                 do i=1,nt 
                    do k=1,3
                       rh(k,i)=rot2(k)*rh(k,i)
                    end do
                    call get_type(mh(i),atom)
                    write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rh(k,i),k=1,3)
                 end do
              end if
           else
              dd=dd1
              if(debug.eq."T") THEN
                 do i=1,nl 
                    do k=1,3
                       rl(k,i)=rot2(k)*rl(k,i)
                    end do
                    call get_type(ml(i),atom)
                    write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rl(k,i),k=1,3)
                 end do
                 do i=1,nt 
                    do k=1,3
                       rh(k,i)=rot2(k)*rh(k,i)
                    end do
                    call get_type(mh(i),atom)
                    write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rh(k,i),k=1,3)
                 end do
              end if
           end if
        ELSE ! close max loop
           if(debug.eq."T") THEN
              do i=1,nl 
                 call get_type(ml(i),atom)
                 write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rl(k,i),k=1,3)
              end do
              do i=1,nt 
                 call get_type(mh(i),atom)
                 write(67,'(A4,1x,3f9.3,5x,3f10.3)') atom,(rh(k,i),k=1,3)
              end do
           end if
        ENDIF
        write(6,'(A3,3f10.4,2f12.6)') "x ",dd
        ddold=dd
     end if ! close rep if 
  end do ! close conf loop
1 stop

end program volume


subroutine inertia(mass,r,n,t) 
  implicit none
  integer n
  real*8 mass(n),r(3,n),t(3,3)
  !local 
  real*8 rr(n),delta(3,3)
  integer i,j,k
  rr=0.d0
  delta=0
  do i=1,3
     delta(i,i)=1.D0
  end do
  do k=1,n
     do j=1,3
        rr(k)=rr(k)+r(j,k)**2
     end do
  end do
  t=0.d0
  do i=1,3
     do j=i,3
        do k=1,n
           t(i,j) = t(i,j) + mass(k)*( delta(i,j)*rr(k) - r(i,k)*r(j,k) )
        end do
     end do
  end do
  do i=1,3
     do j=i,3
           t(j,i) = t(i,j)
     end do
  end do

end subroutine inertia

subroutine rotate(x,rot)
  implicit none
  REAL*8 x(3),new(3),ROT(3,3),A,B,C
  INTEGER I,K
  DO I=1,3
     NEW(i)=0.d0
     DO K=1,3
        NEW(I)=NEW(I)+ROT(k,i)*x(k)
     END DO
  end do
  x=new  ! overwrite old x
end subroutine rotate

subroutine get_mass(atom,mass) 
  implicit none
  character*4 atom
  real*8 mass
  if(atom(1:1).eq."C") mass=12.0 
  if(atom(1:1).eq."N") mass=14.0 
  if(atom(1:1).eq."O") mass=16.0 
  if(atom(1:1).eq."H") mass=1.0 
  if(atom(1:1).eq."F") mass=19.0 
  if(atom(1:1).eq."I") mass=126.0 
  if(atom(1:1).eq."P") mass=31.0 
  if(atom(1:2).eq."CL") mass=35.0 
  if(atom(1:2).eq."BR") mass=79.0 
end subroutine get_mass

subroutine get_type(mass,atom) 
  implicit none
  character*4 atom
  real*8 mass
  if(mass.eq.12.0 )  atom="C"
  if(mass.eq.14.0 )  atom="N"
  if(mass.eq.16.0 )  atom="O"
  if(mass.eq.1.0 )   atom="H"
  if(mass.eq.19.0 )  atom="F"
  if(mass.eq.126.0 ) atom="I"
  if(mass.eq.31.0 )  atom="P"
  if( mass.eq.35.0 ) atom="CL"
  if(mass.eq.79.0 )  atom="BR"
end subroutine get_type


subroutine Jacobi(a,x,abserr,n)
!===========================================================
! Evaluate eigenvalues and eigenvectors
! of a real symmetric matrix a(n,n): a*x = lambda*x 
! method: Jacoby method for symmetric matrices 
! Alex G. (December 2009)
!-----------------------------------------------------------
! input ...
! a(n,n) - array of coefficients for matrix A
! n      - number of equations
! abserr - abs tolerance [sum of (off-diagonal elements)^2]
! output ...
! a(i,i) - eigenvalues
! x(i,j) - eigenvectors
! comments ...
!===========================================================
implicit none
integer i, j, k, n
double precision a(n,n),x(n,n),arr(n)
double precision abserr, b2, bar
double precision beta, coeff, c, s, cs, sc

! initialize x(i,j)=0, x(i,i)=1
! *** the array operation x=0.0 is specific for Fortran 90/95
x = 0.0
do i=1,n
  x(i,i) = 1.0
end do

! find the sum of all off-diagonal elements (squared)
b2 = 0.0
do i=1,n
  do j=1,n
    if (i.ne.j) b2 = b2 + a(i,j)**2
  end do
end do

if (b2 <= abserr) return

! average for off-diagonal elements /2
bar = 0.5*b2/float(n*n)

do while (b2.gt.abserr)
  do i=1,n-1
    do j=i+1,n
      if (a(j,i)**2 <= bar) cycle  ! do not touch small elements
      b2 = b2 - 2.0*a(j,i)**2
      bar = 0.5*b2/float(n*n)
! calculate coefficient c and s for Givens matrix
      beta = (a(j,j)-a(i,i))/(2.0*a(j,i))
      coeff = 0.5*beta/sqrt(1.0+beta**2)
      s = sqrt(max(0.5+coeff,0.0))
      c = sqrt(max(0.5-coeff,0.0))
! recalculate rows i and j
      do k=1,n
        cs =  c*a(i,k)+s*a(j,k)
        sc = -s*a(i,k)+c*a(j,k)
        a(i,k) = cs
        a(j,k) = sc
      end do
! new matrix a_{k+1} from a_{k}, and eigenvectors 
      do k=1,n
        cs =  c*a(k,i)+s*a(k,j)
        sc = -s*a(k,i)+c*a(k,j)
        a(k,i) = cs
        a(k,j) = sc
        cs =  c*x(k,i)+s*x(k,j)
        sc = -s*x(k,i)+c*x(k,j)
        x(k,i) = cs
        x(k,j) = sc
      end do
    end do
  end do
end do
! now sort 
do i=1,n
   arr(i)=a(i,i) 
end do
call piksr2(n,arr,x)
do i=1,n
   a(i,i)=arr(i)
end do
return
end
SUBROUTINE piksr2(n,arr,brr)
  implicit none
  INTEGER n
  REAL*8 arr(n),brr(n,n)
  !Sorts an array arr(1:n) into ascending numerical order, by straight insertion, while making
  !the corresponding rearrangement of the array brr(1:n,*).
  INTEGER i,j,k
  REAL*8 a,b(n)
  do j=2,n ! Pick out each element in turn.
     a=arr(j)
     do k=1,n
        b(k)=brr(k,j)
     end do
     do i=j-1,1,-1 ! Look for the place to insert it.
        if(arr(i).le.a) goto 10
        arr(i+1)=arr(i)
        do k=1,n
           brr(k,i+1)=brr(k,i)
        end do
     enddo
     i=0
10   arr(i+1)=a ! Insert it.
     do k=1,n
        brr(k,i+1)=b(k)
     end do
  enddo
  return
END SUBROUTINE piksr2



