program rmsd_pdb
  ! ----------------------------------------------------------------------
  !  compute RMS of atom-atom Distances between corresponding atoms
  !      of two structures (or a subset), after minimization
  !
  !  fixed names for input files: 
  !
  !  "p1.pdb" = structure n. 1 
  !  "r1.pdb" = refernce  n. 1 (subset of atoms to overlap in 1)
  !  "s1.pdb" = subset    n. 1 (subset of atoms for rmsd in 1)
  !  "p2.pdb" = structure n. 2 
  !  "r2.pdb" = reference n. 2 (subset of atoms to overlap in 2)
  !  "s2.pdb" = subset    n. 2 (subset of atoms for rmsd in 2)
  !
  !  - only atom indexes are read from r1.pdb, r2.pdb, s1.pdb, s2.pdb
  !  - number of atoms in r2.pdb must be = number of atoms in r1.pdb
  !  - number of atoms in s2.pdb must be = number of atoms in s1.pdb
  !  - atom indexes in px.pdb MUST be contiguous (no 1,2,32,43,...)
  ! ----------------------------------------------------------------------

  use pdb
  implicit  none

  integer                  :: i,j,k,l
  integer                  :: err
  integer                  :: fp1,fp2,fr1,fr2,fro,fs1,fs2 ! units
  integer                  :: unit_counter=10
  integer                  :: n1,n2
  integer                  :: p1_nato,p2_nato,r_nato,s_nato,r2_nato,s2_nato ! number of atoms
  real*8,      allocatable :: p1(:,:),p2(:,:),r1(:,:),r2(:,:),s1(:,:),s2(:,:) ! coordinate matrix (3 x nato)
  type(label), allocatable :: r1_label(:),r2_label(:),p1_label(:),p2_label(:),&
                              s1_label(:),s2_label(:) ! pdb label array (nato)
  real*8,      allocatable :: r(:,:),r0(:,:),rfit(:,:) 
  real*8                   :: dd ! rmsd value
  real*8                   :: d(3,3) ! rotation matrix
  real*8                   :: tr
  logical                  :: endofpdb

  
  ! open unit, allocate memory, read pdb and close unit
  ! for r1,s1,r2,s2 files
  
  !----------- r1.pdb
  call open_pdb(unit_counter,fr1,"r1.pdb","OLD",r_nato)

  allocate(r1(3,r_nato),r1_label(r_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (r1)"
   
  call read_pdb(r_nato,fr1, r1, r1_label,endofpdb)
  
  close(fr1)

  !----------- r2.pdb
  call open_pdb(unit_counter,fr2,"r2.pdb","OLD",r2_nato)
  
  if (r2_nato .ne. r_nato) stop "r2_nato .ne. r1_nato"
  
  allocate(r2(3,r_nato),r2_label(r_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (r2)"
  
  call read_pdb(r_nato,fr2, r2, r2_label,endofpdb)
  
  close(fr2)

  !----------- s1.pdb
  call open_pdb(unit_counter,fs1,"s1.pdb","OLD",s_nato)

  allocate(s1(3,s_nato),s1_label(s_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (s1)"
  
  call read_pdb(s_nato,fs1, s1, s1_label,endofpdb)
  
  close(fs1)

  !----------- s2.pdb
  call open_pdb(unit_counter,fs2,"s2.pdb","OLD",s2_nato)
  
  if (s2_nato .ne. s_nato) stop "s2_nato .ne. s_nato"
  
  allocate(s2(3,s_nato),s2_label(s_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (s2)"
  
  call read_pdb(s_nato,fs2, s2, s2_label,endofpdb)
  
  close(fs2)

  ! allocate r,r0,rfit work arrays

  allocate(r(3,r_nato),r0(3,r_nato),rfit(3,r_nato),&
       stat = err)
  if(err /= 0) stop "error in allocating memory (r,r0,rfit)"
  
  ! finally open and read p1 and p2 files
  
  call open_pdb(unit_counter,fp1,"p1.pdb","OLD",p1_nato)
  
  call open_pdb(unit_counter,fp2,"p2.pdb","OLD",p2_nato)

  if (r_nato .gt. p1_nato) stop "r_nato .gt. p1_nato"
  if (s_nato .gt. p1_nato) stop "s_nato .gt. p1_nato"
  if (r_nato .gt. p2_nato) stop "r_nato .gt. p2_nato"
  if (s_nato .gt. p2_nato) stop "s_nato .gt. p2_nato"
  
  allocate(p1(3,p1_nato),p1_label(p1_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (p1)"
  
  allocate(p2(3,p2_nato),p2_label(p2_nato),stat = err)
  if(err /= 0) stop "error in allocating memory (p2)"
  
  n1 = 0
  
  pdb1_loop: do 
     
     call read_pdb(p1_nato,fp1, p1, p1_label,endofpdb)

     if(endofpdb) exit pdb1_loop

     !!debug
     
     !!call print_pdb(99,p1,p1_nato,p1_label)
     
     n1 = n1 + 1
     
     n2 = 0
     
     pdb2_loop: do 
        
        call read_pdb(p2_nato,fp2, p2, p2_label,endofpdb)

        if(endofpdb) exit pdb2_loop
        
        n2 = n2 + 1
        
        ! open rotated structure unit
        
        ! call open_pdb(unit_counter,fro,"ro.pdb","REP",p2_nato)
        
        ! build r,r0,s1,s2
        do l = 1,r_nato
           do k = 1,p1_nato
              if(p1_label(k)%serial == r1_label(l)%serial) r(:,l)  = p1(:,k)
           end do
           do k = 1,p2_nato
              if(p2_label(k)%serial == r2_label(l)%serial) r0(:,l)  = p2(:,k)
           end do
        end do
        do l = 1,s_nato
           do k = 1,p1_nato
              if(p1_label(k)%serial == s1_label(l)%serial) s1(:,l) = p1(:,k)
           end do
           do k = 1,p2_nato
              if(p2_label(k)%serial == s2_label(l)%serial) s2(:,l) = p2(:,k)
           end do
        end do
           
        ! ----- engine 
        call rmsd(r_nato,r,r0,dd,rfit,d)
        ! ------------
        
        ! rotate p2 and s2
        
        do j = 1,3
           tr = sum(r0(j,:)) / real(r_nato)
           p2(j,:) = p2(j,:) - tr
           s2(j,:) = s2(j,:) - tr
        end do
        p2 = matmul(d,p2)
        s2 = matmul(d,s2)
        do j = 1,3
           tr = sum(r(j,:)) / real(r_nato)
           p2(j,:) = p2(j,:) + tr
           s2(j,:) = s2(j,:) + tr
        end do
        
        ! print rotated p2 structure
        
        ! call print_pdb(fro, p2, p2_nato, p2_label)
        
        ! compute rmsd 
        
        dd = sqrt(sum((s2-s1)**2) / real(s_nato))
        
        write(6,*) n1,"          <->",n2,dd
        
     end do pdb2_loop

     rewind(fp2)
    
  end do pdb1_loop

end program rmsd_pdb
