#! /bin/bash

# Compute RMSD of atoms in structure[s] 1 from atoms in structure[s] 2
#   including only atoms listed in reference files
#
# syntax: $0 [{-1 R1 -2 R2 | -r R }] [{-a S1 -b S2 | -s S }] "P1 ..." "P2 ..." 
#
#   this is just a shell wrapper for `rmsd_pdb' allowing any file name
#   and multiple PDB files 
#
#  P1 = structure n. 1 
#  R1 = reference n. 1 (subset of atoms for alignment in 1)
#  S1 = subset    n. 1 (subset of atoms for rmsd calc in 1)
#  P2 = structure n. 2 
#  R2 = reference n. 2 (subset of atoms for alignment in 2)
#  S2 = subset    n. 2 (subset of atoms for rmsd calc in 2)
#  R  =  alignment subset for both
#  S  =  rmsd calc subset for both
#
#  - if -1 or -2 is omitted, all atoms are included
#  - only atom indexes are read from R1, R2, S1, S2
#  - first atom in R1 is compared to first atom in R2, etc.
#  - number of atoms in R2 (S2)  must be = number of atoms in R1 (S1)
#  - last record in all files must be "END" or "TER"
#
#
# N.B.: quotes are REQUIRED around LISTS of files, i.e.
#  rmsd.sh start-structure.pdb "PAR00*/out.0000" 
#

# ----------------------------------------------------------------------
# help: copy first comments block in this file
# ----------------------------------------------------------------------
function help() {
name=`basename "$0"`;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------
FILES="r1.pdb r2.pdb p1.pdb p2.pdb s1.pdb s2.pdb"
###BIN=`dirname "$0"`/src;
BIN=`dirname "$0"`;
rm $FILES &> /dev/null;

# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "1:2:a:b:r:s:h-" Option
do
  case $Option in
      1     )  R1=$OPTARG;;
      2     )  R2=$OPTARG;;
      r     )  R=$OPTARG ;;
      a     )  S1=$OPTARG;;
      b     )  S2=$OPTARG;;
      s     )  S=$OPTARG;;
      *     )    help;;   # help: copy first comments block
  esac
done

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

if [[ "$R" != "" && ( "$R1" != "" || "$R2" != "" ) ]] ; then 
    echo "*********************************************************"
    echo "** WARNING: -r and -12 options are mutually exclusive!!"; 
    echo "   type '`basename $0` -h' for help"
    echo "*********************************************************"
    exit; 
fi  
if [[ "$S" != "" && ( "$S1" != "" || "$S2" != "" ) ]] ; then 
    echo "*********************************************************"
    echo "** WARNING: -s and -ab options are mutually exclusive!!"; 
    echo "   type '`basename $0` -h' for help"
    echo "*********************************************************"
    exit; 
fi  

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
if [[ $# -eq 0 ]] ; then 
    help; 
    exit;     
elif [[ $# -eq 1 ]] ; then 
    P1=$1
    P2=$1
elif [[ $# -eq 2 ]] ; then 
    P1=$1;
    P2=$2;
else
    help; 
    exit;     
fi


# ----------------------------------------------------------------------
# read user's configuration
# ----------------------------------------------------------------------

# ----------------------------------------------------------------------
# final parameters setting
# ----------------------------------------------------------------------


if [[ "$R" != "" ]] ; then  
    R1=$R;
    R2=$R;
else
    if [[ "$R1" == "" ]] ; then  
	R1=${P1%' '*};
    fi
    if [[ "$R2" == "" ]] ; then  
	R2=${P2%' '*};
    fi
fi

if [[ "$S" != "" ]] ; then  
    S1=$S;
    S2=$S;
else
    if [[ "$S1" == "" ]] ; then  
	S1=$R1;
    fi
    if [[ "$S2" == "" ]] ; then  
	S2=$R2;
    fi
fi

ln -sf $R1 r1.pdb;
ln -sf $R2 r2.pdb;
ln -sf $S1 s1.pdb;
ln -sf $S2 s2.pdb;

# ----------------------------------------------------------------------
# engine
# ----------------------------------------------------------------------

for i in $P1; do
    ln -sf $i p1.pdb; 
    for j in $P2; do 
	ln -sf $j p2.pdb; 
	if [[ "$i" == "$j" ]]; then 
	    # continue; 
	    rm p2.pdb &&  cp $i p2.pdb;
	fi;
	printf "# %s [%s] [%s] <=> %s [%s] [%s]" `basename $i` $R1 $S1 `basename $j` $R2 $S2;
	echo;
	${BIN}/rmsd_pdb;
    done
done

