module precision 
  
  integer, parameter :: kr4 = selected_real_kind(6,37)      
  integer, parameter :: kr8 = selected_real_kind(15,307)  
  integer, parameter :: krlong = selected_real_kind(18,400)  
  integer, parameter :: kr16 = selected_real_kind(30,1000) 
  integer, parameter :: ki4 = selected_int_kind(9)           
  integer, parameter :: ki8 = selected_int_kind(18)          
  integer, parameter :: kc4 = kr4                            
  integer, parameter :: kc8 = kr8                            
  ! generic kinds
  integer, parameter :: ki=ki4,kr=krlong,kc=kc8
  
end module precision

!**********************************************************************
!**********************************************************************


module strings

use precision

!**********************************************************************

contains

!**********************************************************************

subroutine readline(nunitr,line,ios)

! Reads line from unit=nunitr, ignoring blank lines
! and deleting comments beginning with an exclamation point(!)

character (len=*):: line

do  
  read(nunitr,'(a)', iostat=ios) line      ! read input line
  if(ios /= 0) return
  line=adjustl(line)
  ipos=index(line,'!')
  if(ipos == 1) cycle
  if(ipos /= 0) line=line(:ipos-1)
  if(len_trim(line) /= 0) exit
end do
return

end subroutine readline

!**********************************************************************

subroutine compact(str)

! Converts multiple spaces and tabs to single spaces; deletes control characters;
! removes initial spaces.

character(len=*):: str
character(len=1):: ch
character(len=len_trim(str)):: outstr

str=adjustl(str)
lenstr=len_trim(str)
outstr=' '
isp=0
k=0

do i=1,lenstr
  ch=str(i:i)
  ich=iachar(ch)
  
  select case(ich)
  
    case(9,32)     ! space or tab character
      if(isp==0) then
        k=k+1
        outstr(k:k)=' '
      end if
      isp=1
      
    case(33:)      ! not a space, quote, or control character
      k=k+1
      outstr(k:k)=ch
      isp=0
      
  end select
  
end do

str=adjustl(outstr)

end subroutine compact

!**********************************************************************

subroutine split(str,delims,before,sep)

! Routine finds the first instance of a character from 'delims' in the
! the string 'str'. The characters before the found delimiter are
! output in 'before'. The characters after the found delimiter are
! output in 'str'. The optional output character 'sep' contains the 
! found delimiter. A delimiter in 'str' is treated like an ordinary 
! character if it is preceded by a backslash (\). If the backslash 
! character is desired in 'str', then precede it with another backslash.

character(len=*) :: str,delims,before
character,optional :: sep
logical :: pres
character :: ch,cha

pres=present(sep)
str=adjustl(str)
call compact(str)
lenstr=len_trim(str)
if(lenstr == 0) return        ! string str is empty
k=0
ibsl=0                        ! backslash initially inactive
before=' '
do i=1,lenstr
   ch=str(i:i)
   if(ibsl == 1) then          ! backslash active
      k=k+1
      before(k:k)=ch
      ibsl=0
      cycle
   end if
   if(ch == '\') then          ! backslash with backslash inactive '\'
      k=k+1
      before(k:k)=ch
      ibsl=1
      cycle
   end if
   ipos=index(delims,ch)         
   if(ipos == 0) then          ! character is not a delimiter
      k=k+1
      before(k:k)=ch
      cycle
   end if
   if(ch /= ' ') then          ! character is a delimiter that is not a space
      str=str(i+1:)
      if(pres) sep=ch
      exit
   end if
   cha=str(i+1:i+1)            ! character is a space delimiter
   iposa=index(delims,cha)
   if(iposa > 0) then          ! next character is a delimiter
      str=str(i+2:)
      if(pres) sep=cha
      exit
   else
      str=str(i+1:)
      if(pres) sep=ch
      exit
   end if
end do
if(i >= lenstr) str=''
str=adjustl(str)              ! remove initial spaces
return

end subroutine split

!**********************************************************************

subroutine nargsline(nunitr,line,delims,nargs,ios)
  
  ! returns the number of arguments in a line from unit=nunitr in the integer output variable nargs
  ! based on the delimiters contained in the string delims, ignoring blank lines
  ! and deleting comments beginning with an exclamation point(!)
  ! do not move on unit
  
  implicit none
  integer(ki),intent(in) :: nunitr
  character(len=*),intent(out) :: line
  character(len=*),intent(in) :: delims
  integer(ki),intent(out) :: nargs
  integer(ki),intent(out) :: ios
  ! local variables
  character(len=len(line)) :: line1
  character(len=len(line)) :: before
  integer(ki) :: lenstr
  
  call readline(nunitr,line,ios)
  if(ios /= 0) return
  
  line1=line
  call compact(line1)
  nargs=0
  lenstr=len_trim(line1)
  if(lenstr==0) return
  do
     if(len_trim(line1) == 0) exit
     nargs=nargs+1
     call split(line1,delims,before)
  end do
  
end subroutine nargsline

!**********************************************************************


end module strings                                                   

!**********************************************************************
!**********************************************************************

program mbar
  ! multiple bennett acceptance ratio (MBAR) algorithm 

  use precision
  use strings

!**********************************************************************
  
  implicit none
  
  integer(ki) :: i,j,c,s,iter
  integer(ki) :: err
  character(len=1280) :: line
  integer(ki) :: nargs
  character(len=128),allocatable :: args(:)
  integer(ki),parameter :: upar=11
  integer(ki),parameter :: udata=12
  character(len=128) :: fpar 
  character(len=128) :: fdata
  real(kr) :: x
  integer(ki) :: np ! Number of Parameters per state
  integer(ki) :: ns ! Number of States
  integer(ki) :: nc ! Number of Configurations 
  real(kr),allocatable :: parms(:,:)
  real(kr),allocatable :: pots(:,:)
  integer(ki),allocatable :: state(:)
  integer(ki),allocatable :: ncs(:)
  real(kr),allocatable :: z(:),z0(:)
  real(kr),allocatable :: q(:,:)
  real(kr),allocatable :: d(:)
  real(kr),parameter :: eps=0.01
  real(kr) :: diff
  !!
  character(len=1280) :: string
  real(kr) :: w
  logical :: conv = .false.
  logical :: fileread = .false.
  real(kr) :: minw=10000000.0
  
  ! print command line on standard error 
  call get_command(line)
  write(0,*) trim(line)
  
  !-------------- read arguments 
  call readargs(args,nargs,err)
  if(err /= 0) THEN 
     write(*,*) "Error reading arguments "
     write(*,*) "syntax:  mbar PARAMETER_FILE DATA_FILE"
     stop 
  END IF
  if(nargs /= 2) THEN 
     write(*,*) "Error reading arguments"
     write(*,*) "syntax:  mbar PARAMETER_FILE DATA_FILE"
     stop
  END IF
  fpar = args(1)
  fdata = args(2)
  write(0,*) "parameter file: ", trim(fpar)
  write(0,*) "data file: ", trim(fdata)

  !------------- open units
  open(unit=upar,file=trim(fpar),status='OLD',ERR=1001)
  open(unit=udata,file=trim(fdata),status='OLD',ERR=1002)
  fileread=.true. 

  !------------- get np,ns,nc
  ns = 0
  np = 0
  do 
     i = np
     call nargsline(upar,line,' ',np,err) 
     if(i /= 0 .and. np /= i) stop "check the parameters file"
     if(err < 0) exit
     ns = ns + 1
  end do
  rewind(upar)
  nc = 0
  do 
     read(udata,*,iostat=err) x
     if(err /= 0) then
        if(err < 0) then 
           exit
        else
           stop "error in data file" 
        end if
     end if
     nc = nc + 1
  end do
  rewind(udata)

  !------------- print headers
  write(0,*) "number of states: ", ns
  write(0,*) "number of parameters per state: ", np
  write(0,*) "number of config. : ", nc

  !------------- allocate memory
  allocate(parms(ns,np),stat=err)
  if(err /= 0) stop "error allocating parms"
  allocate(state(nc),pots(nc,np),stat=err)
  if(err /= 0) stop "error allocating state,pots"
  allocate(ncs(ns),stat=err)
  if(err /= 0) stop "error allocating ncs"
  allocate(z(ns),z0(ns),stat=err)
  if(err /= 0) stop "error allocating z,z0"
  allocate(q(nc,ns),d(nc),stat=err)
  if(err /= 0) stop "error allocating q,d"

  !------------- read 
  do i = 1,ns
     read(upar,*,iostat=err) parms(i,:)
     if(err /= 0) stop "error in parameters file"
  end do
  do i = 1,nc
     read(udata,*,iostat=err) state(i),pots(i,:)
     if(err /= 0) stop "error in data file"
     if(state(i) > ns) stop "wrong state number"
  end do

  !------------- print
!  do i = 1,ns
!     write(0,*) parms(i,:)
!  end do
!  write(*,*) 
!  do i = 1,nc
!     write(0,*) state(i),pots(i,:)
!  end do

  ! compute ncs
  ncs = 0
  do i = 1,nc
     ncs(state(i)) = ncs(state(i)) + 1
  end do


  ! compute exp once for all
  ! POTS nc x np, parms ns x np, q nc x ns
  q = matmul(pots,transpose(parms))

  ! remove max abosolute value
  
  do i = 1,nc
     if(maxval(abs(q(i,:))) > maxval(q(i,:)))then 
        q(i,:) = q(i,:) - minval(q(i,:))
     else
        q(i,:) = q(i,:) - maxval(q(i,:))
     end if
  end do

  ! take exp
  q = exp(-q)

!  z = 1.0
  ! compute initial guess
  z = 0.0
  do c = 1,nc
     z(state(c)) = z(state(c)) + sum(pots(c,:))
  end do
  z = z / ncs
  z = z/z(1)
  iter = 0
  inner_loop: do 
     iter = iter + 1
     do j = 1,ns
        if(abs(z(j)) < tiny(x)) then 
           write(0,*) "zero value of z in state ",j
           !           z(j) = z(j) + tiny(x) 
           stop
        end if
     end do

     z0 = z
     z = 0.0
     c_loop: do c = 1,nc
        d(c) = sum(ncs*q(c,:)/z0)
!        write(*,*) iter, c, d(c), z(1),z(ns)
        s_loop: do s = 1,ns
           ! compute weights for conf. c in state s
           w = q(c,s) / d(c)
           z(s) = z(s) + w
           if(w < minw) minw = w
           if(conv .and. s == 1) write(*,*) w
        end do s_loop
     end do c_loop
     z = z/z(1)
     if(iter > 1) then 
        if(conv) exit inner_loop
        ! check convergence
        diff = maxval(abs(log(z)-log(z0)))
        write(0,'(a,i5,3g15.5)') 'diff', iter, diff
        if(diff < eps) conv = .true. ! converged: print weights in next iteration and exit
     end if
  end do inner_loop

  write(0,*) "goodbye!", z(ns), minw
  if(fileread) stop
1001 write(*,*) "Parameter file '", trim(fpar),"' not found"
1002 write(*,*) "Data File '", trim(fdata),"' not found"
contains 

!**********************************************************************
  
  subroutine readargs(args,nargs,err)
    use precision
    implicit none
    character(len=*),allocatable,intent(out) :: args(:)
    integer(ki),intent(out) :: nargs
    integer(ki),intent(out) :: err 
    ! local variables
    integer(ki) :: i
    
    err = 0
    
    ! get arguments number
    
    nargs = command_argument_count()

    allocate(args(nargs),stat=err)
    
    ! read arguments
    
    do i = 1,nargs
       call get_command_argument(i, args(i))
    end do
    
  end subroutine readargs

!**********************************************************************
  
end program mbar

