!-----------------------------------------------------------------------------
!
!dcd2dpb 
!
!NAME
!       dcd2pdb - convert a dcd file into a pdb file.
!
!SYNOPSIS
!       dcd2pdb -pdb <pdb ref. file> -dcd <dcd file> [-f <list-of-frames file>].
!
!OVERVIEW
!       The dcd2pdb converts a trajectory file from DCD to PDB format, using 
!       a reference pdb file for labels. 
!
!OPTIONS
!       -f, --frames <list-of-frames file>
!           output only those frames contained in the list-of-frames file.
!           the frames file should contain a single column of integers.
!       -h, --help
!           print this man
!
!AUTHOR
!       Written by Simone Marsili.
!
!-----------------------------------------------------------------------------

!-----------------------------------------------------------------------------

module pdb
  type :: label ! define new type label
     character*6 :: recrd
     integer :: serial
     character*4 :: name
     character*1 :: altloc
     character*3 :: resname
     character*1 :: chainid
     integer :: resseq
     character*1 :: icode
  end type label
  
contains
  
  subroutine open_pdb(unit_counter,unit_number,file_name,file_status,nato)
    ! open unit unit_number connected to the pdb file file_name
    ! count the number of atoms in the pdb 
    implicit none
    integer,intent(inout) :: unit_counter
    integer, intent(out) :: unit_number
    integer, intent(out) :: nato
    character*(*), intent(in) :: file_name
    character*3, intent(in) :: file_status
    integer :: err
    character*80 :: line
    
    unit_counter = unit_counter + 1
    
    unit_number = unit_counter
    
    select case(file_status)
    case("OLD")
       open(unit=unit_number,file=file_name,status='OLD',iostat=err)
       if(err /= 0) then 
          write(0,*) "error opening unit ", unit_number, "pdbfile ", file_name
          stop
       end if
       nato = 0
       read_loop: do 
          read(unit_number,'(A80)',iostat=err) line
          if(err < 0) exit read_loop
          select case(line(1:6))
          case("ATOM  ","HETATM") 
             nato = nato + 1
          case("END   ","TER   ","ENDMDL") 
             exit  read_loop
          case default 
             cycle read_loop
          end select
       enddo read_loop
       rewind(unit_number)
    case("NEW")
       open(unit=unit_number,file=file_name,status='NEW',iostat=err)
    case("REP")
       open(unit=unit_number,file=file_name,status='REPLACE',iostat=err)
    case("SCR")
       open(unit=unit_number,status='SCRATCH',iostat=err)
    case("UNK")
       open(unit=unit_number,file=file_name,status='UNKNOWN',iostat=err)
    end select
    
    if(nato == 0) then 
       write(0,*) "file ",file_name,"contains 0 atoms"
       stop
    end if
    
  end subroutine open_pdb
  
  
  subroutine read_pdb(nato,unit,r,pdblabel,endofpdb)
    implicit none
    integer,intent(in)     :: nato         ! number of atoms
    integer,intent(in)     :: unit         ! pdb unit
    real(4),intent(out)   :: r(:,:)       ! coordinates matrix
    type(label),intent(out):: pdblabel(:)  ! pdb label array
    logical,intent(out)    :: endofpdb          ! endofpdb
    ! local variables
    integer :: n            ! number of atoms
    integer            :: i,j
    integer            :: err       
    character*80           :: line
    
    endofpdb = .false.
    n = 0
    read_loop: do        
       read(unit,'(A80)',iostat=err) line
       if(err < 0) then 
          endofpdb = .true.
          exit read_loop
       end if
       select case(line(1:3))
       case("ATO","HET") 
          n=n+1
          read(line,1,iostat=err) pdblabel(n),(r(j,n),j=1,3)
          if(err > 0) then 
             write(0,*) 'error while reading line ',n,'of unit',unit
             stop
          end if
          pdblabel(n)%name = adjustl(pdblabel(n)%name)
       case("END","TER")  
          if(n == 0) cycle read_loop
          exit  read_loop
       case default 
          cycle read_loop
       end select
    enddo read_loop
    
1   format(a6,i5,1x,a4,a1,a3,1x,a1,i4,a1,3x,3f8.3,2f6.2)
    
  end subroutine read_pdb

  subroutine print_pdb(box,unit,r,n,pdblabel)
    implicit none
    real(8),intent(in) :: box(6)
    integer,intent(in)     :: unit         ! pdb unit
    real(4),intent(in)    :: r(:,:)       ! coordinates matrix
    integer,intent(in)     :: n            ! number of atoms
    type(label),intent(in) :: pdblabel(:)  ! pdb label array
    ! local variables
    integer            :: i,j
    integer            :: err       

    write(unit,'(a,1x,6f15.6)') 'REMARK   a, b, c, alpha, beta, gamma:', box(:)
    do i = 1,n
       write(unit,1) pdblabel(i), (r(j,i),j=1,3)
    end do
    write(unit,'(a3)') 'END'
    
1   format(a6,i5,1x,a4,a1,a3,1x,a1,i4,a1,3x,3f8.3,2f6.2)
    
  end subroutine print_pdb

end module pdb

!-----------------------------------------------------------------------------

!-----------------------------------------------------------------------------

module dcd
  implicit none
  integer :: max_dumps=10000
  integer :: dcd_frame = 0
  
contains
  
  subroutine read_dcd(unit,nato,box,x,y,z,eof)
    implicit none
    integer,intent(in) :: unit
    integer,intent(in) :: nato
    real(8),intent(out) :: box(6)
    real(4),intent(out) :: x(nato)
    real(4),intent(out) :: y(nato)
    real(4),intent(out) :: z(nato)
    logical,intent(out) :: eof
    integer :: err

    dcd_frame = dcd_frame + 1
    
    if(dcd_frame == 1) call read_dcd_header(unit,nato)
    
    read(unit,iostat=err) box
    if(err < 0) then 
       eof = .true.
       return
    end if
    read(unit) x
    read(unit) y
    read(unit) z

  end subroutine read_dcd
  
  subroutine read_dcd_header(unit,nato)
    implicit none
    integer,intent(in) :: unit
    integer,intent(in) :: nato
    character*4 :: hdr
    integer :: icntrl(20), nstr
    integer,parameter :: max_dumps=1000
    integer :: nat1

    
    read(unit) hdr, icntrl
    read(unit) nstr
    read(unit) nat1

  end subroutine read_dcd_header

end module dcd

!-----------------------------------------------------------------------------

!-----------------------------------------------------------------------------

program dcd2pdb
  use pdb
  use dcd
  implicit none
  integer :: err
  integer :: updb=0,udcd=0,uframes=0
  integer :: nato
  integer :: i,t
  integer :: unit_counter = 10
  real(4),allocatable :: r(:,:),x(:),y(:),z(:)
  type(label),allocatable :: pdblab(:)
  logical :: eof=.false.
  character*128 :: line
  real(8) :: box(6) = 0.0
  integer :: nargs,iarg
  character(90) :: arg
  logical :: exist=.false.
  integer :: nframes,frame
  integer,allocatable :: frames(:)

  ! read line arguments
  nargs = command_argument_count()
  iarg = 0
  do 
     if(iarg == nargs) exit
     iarg = iarg + 1
     call get_command_argument(iarg,arg)
     select case(trim(arg))
     case('-pdb')
        ! read pdb filename 
        iarg = iarg + 1
        call get_command_argument(iarg,arg)
        inquire(FILE=trim(arg), EXIST=exist)
        if(exist) then 
           call open_pdb(unit_counter,updb,trim(arg),'OLD',nato)
        else
           write(0,*) 'cant find pdb file'
           stop
        end if
     case('-dcd')
        ! read dcd filename 
        iarg = iarg + 1
        call get_command_argument(iarg,arg)
        unit_counter = unit_counter + 1
        udcd = unit_counter
        open(unit=udcd,file=trim(arg),form='UNFORMATTED')
     case('-f','--frames')
        ! read frames list filename 
        iarg = iarg + 1
        call get_command_argument(iarg,arg)
        inquire(FILE=trim(arg), EXIST=exist)
        unit_counter = unit_counter + 1
        uframes = unit_counter
        if(exist) then 
           open(unit=uframes,file=trim(arg),form='FORMATTED')
        else
           write(0,*) 'cant find frames list file'
           stop
        end if
     case('-h','--help','--man')
        write(0,100) 
     case default
        write(0,*) 'unknown option'
        write(0,100)
        stop
     end select
  end do

  if(updb == 0 .or. udcd == 0) then 
     write(0,100)
     stop
  end if

  ! allocate memory for pdb variables
  allocate(r(3,nato),pdblab(nato),stat=err)
  allocate(x(nato),y(nato),z(nato),stat=err)
  if(err /= 0) stop 'allocation failed for r,pdblab'

  ! read reference pdb file 
  call read_pdb(nato,updb,r,pdblab,eof)

  ! if a frames list to print is given, read it
  
  if(uframes /= 0) then 
     nframes = 0
     do 
        read(uframes,*,iostat=err) frame
        if(err < 0) exit
        nframes = nframes + 1
     end do
     rewind(uframes)
     allocate(frames(nframes),stat=err)
     do i = 1,nframes
        read(uframes,*) frames(i)
     end do
  end if

  
  ! read dcd file
  eof = .false.
  t = 0
  do 
     t = t + 1
     call read_dcd(udcd,nato,box,x,y,z,eof)
     if(eof) exit
     if(uframes == 0) then 
        r(1,:) = x
        r(2,:) = y
        r(3,:) = z
        call print_pdb(box,6,r,nato,pdblab)     
     else
        if(any(frames == t)) then 
           r(1,:) = x
           r(2,:) = y
           r(3,:) = z
           call print_pdb(box,6,r,nato,pdblab)     
        end if
     end if
  end do

100     format (&
             'dcd2dpb                                                                        ' /,&
             '                                                                               ' /,&
             'NAME                                                                           ' /,&  
             '       dcd2pdb - convert a dcd file into a pdb file.                           ' /,&  
             '                                                                               ' /,&  
             'SYNOPSIS                                                                       ' /,&  
             '       dcd2pdb -pdb <pdb ref. file> -dcd <dcd file> [-f <list-of-frames file>].' /,&  
             '                                                                               ' /,&  
             'OVERVIEW                                                                       ' /,&  
             '       dcd2pdb converts a trajectory file from DCD to PDB format, using        ' /,&  
             '       a reference pdb file for labels.                                        ' /,&  
             '                                                                               ' /,&  
             'OPTIONS                                                                        ' /,&  
             '       -f, --frames <list-of-frames file>                                      ' /,&  
             '           output only those frames contained in the list-of-frames file.      ' /,&  
             '           the frames file should contain a single column of integers.         ' /,&  
             '       -h, --help                                                              ' /,&  
             '           print this man                                                      ' /,&  
             '                                                                               ' /,&  
             'AUTHOR                                                                         ' /,&  
             '       Written by Simone Marsili.                                              '   &
             )
  
end program dcd2pdb
