module cv
  use precision
  use constants
  use pdb
  implicit none
  integer(ki) :: ncv
  integer(ki),parameter :: dmax=3 ! max dimensionality for a cv
  type :: cvtype
     character*64 :: name ! cv name 
     integer(ki)  :: num ! cv dimensionality (up to dmax)
     real(kr)     :: val(dmax) ! cv current value
     integer(ki)  :: list(4) ! list of (atomic, resid,...) index
     character*64 :: flag 
  end type cvtype
  type(cvtype), allocatable :: cv_data(:)
  
contains

  subroutine allocate_cv
    implicit none
    integer(ki) :: err
    
    allocate(cv_data(ncv),stat=err)
    
    if(err /= 0) then 
       write(0,*) 'error while allocating cv'
       stop
    end if

  end subroutine allocate_cv

  subroutine comp_cv(nato,r,r_label)
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in) :: r(3,nato)
    type(label), intent(in):: r_label(nato)
    ! local variables
    integer(ki) :: i
    integer(ki) :: err
    real(kr) :: cvv1,cvv2,cvv3
    character*64 :: name
    integer(ki) :: list(4)
    character*64 :: flag
    character*64 :: frm
    integer(ki) :: num

    do i = 1,ncv
       name = cv_data(i)%name
       list = cv_data(i)%list
       flag = cv_data(i)%flag
       num = cv_data(i)%num
       select case(name)
       case('atom_distance')
          call atom_distance(nato,r,list(1),list(2),cvv1)
       case('atom_angle')
          call atom_angle(nato,r,list(1),list(2),list(3),cvv1)
       case('atom_dihedral')
          call atom_dihedral(nato,r,list(1),list(2),list(3),list(4),cvv1)
       case('ramachandran')
          call ramachandran(nato,r,r_label,list(1),cvv1,cvv2)
       case('n_hbonds')
          call n_hbonds(nato,r,r_label,list(1),cvv1)
       case('line_fit')
          call line_fit(r,list(1),list(2),cvv1,cvv2,cvv3)
       case('puckering')
          call puckering(nato,r,r_label,list(1),cvv1,cvv2,cvv3)
       end select
       cv_data(i)%val(1) = cvv1
       cv_data(i)%val(2) = cvv2
       cv_data(i)%val(3) = cvv3
    end do

  end subroutine comp_cv
  
  real(kr) function distance(r1,r2)
    implicit none
    real(kr), intent(in) :: r1(3),r2(3)
    ! local variables
    real(kr) :: d12(3)
    
    d12 = r2 - r1
    
    distance = sqrt(dot_product(d12,d12))
    
  end function distance

  real(kr) function angle(r1,r2,r3)
    implicit none
    real(kr), intent(in) :: r1(3),r2(3),r3(3)
    ! local variables
    real(kr) :: r21(3),r23(3)
    r21 = r1 - r2
    r23 = r3 - r2
    r21 = r21 / distance(r1,r2)
    r23 = r23 / distance(r2,r3)
    
    angle = acos(dot_product(r21,r23)) * 180.0_kr / pi
    
  end function angle

  subroutine plane_normal(r1,r2,r3,n)
    ! the normal has the direction and the verso of the cross product between (r2-r1) x (r3-r1)
    implicit none
    real(kr), intent(in) :: r1(3),r2(3),r3(3)
    real(kr), intent(out)   :: n(3)
    ! local variables
    real(kr)                :: r12(3), r23(3)
    
    r12 = (r2 - r1) / distance(r1,r2)
    r23 = (r3 - r2) / distance(r2,r3)

    CALL cross_product(r12,r23,n)

  end subroutine plane_normal

  
  REAL(kr) FUNCTION dihedral(r1,r2,r3,r4)
    implicit none
    real(kr), intent(in) :: r1(3),r2(3),r3(3),r4(3)
    ! local
    REAL(kr)    :: ccos,ssin
    REAL(kr)    :: r23(3),n123(3),n234(3),nn(3)

    r23 = (r3 - r2) / distance(r2,r3)
    call plane_normal(r1,r2,r3,n123)
    call plane_normal(r2,r3,r4,n234)
    CALL cross_product(n123, n234, nn)

    ccos = DOT_PRODUCT(n123, n234)
    ssin = DOT_PRODUCT(r23, nn)

    dihedral = atan2(ssin, ccos) * 180.0_kr / pi
    
  END FUNCTION dihedral

  subroutine atom_distance(nato,r,at1,at2,d)
    ! returns the distance between atoms at1 and at2
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    integer(ki), intent(in) :: at1,at2
    real(kr), intent(out)   :: d
    ! local variables
    real(kr) :: r1(3),r2(3)
    
    r1 = r(:,at1)
    r2 = r(:,at2)

    d = distance(r2,r1)

  end subroutine atom_distance

  subroutine atom_angle(nato,r,at1,at2,at3,a)
    ! returns the angle between atoms at1-at2-at3
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    integer(ki), intent(in) :: at1,at2,at3
    real(kr), intent(out)   :: a
    ! local variables
    real(kr) :: r1(3),r2(3),r3(3)
    
    r1 = r(:,at1)
    r2 = r(:,at2)
    r3 = r(:,at3)
    
    a = angle(r1,r2,r3)
    
  end subroutine atom_angle

  subroutine atom_dihedral(nato,r,at1,at2,at3,at4,dihed)
    ! returns the dihedral angle between atoms at1-at2-at3-at4
    !  1            4
    !   \          /
    !    \        /
    !     2------3
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    integer(ki), intent(in) :: at1,at2,at3,at4
    real(kr), intent(out)   :: dihed
    ! local variables
    real(kr) :: r1(3),r2(3),r3(3),r4(3)
    
    r1 = r(:,at1)
    r2 = r(:,at2)
    r3 = r(:,at3)
    r4 = r(:,at4)
    
    dihed = dihedral(r1,r2,r3,r4)
    
  end subroutine atom_dihedral
  
  subroutine cross_product(a,b,c)     
    !  returns the right-handed vector cross product C = A x B
    implicit none
    real(kr), intent(in)  :: a(3),b(3)
    real(kr), intent(out) :: c(3)
    
    c(1) = a(2)*b(3) - a(3)*b(2)                                         
    c(2) = a(3)*b(1) - a(1)*b(3)
    c(3) = a(1)*b(2) - a(2)*b(1)
    
  end subroutine cross_product

  subroutine ramachandran(nato,r,r_label,res,phi,psi)
    ! returns the array angles=(phi,psi) of residue r. (no correction for chirality)
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    type(label), intent(in) :: r_label(nato)
    integer(ki), intent(in) :: res
    real(kr),intent(out) :: phi, psi
    ! local variables
    integer(ki) :: i
    integer(ki) :: cm,n,ca,c,np
    integer(ki) :: res0
    character*4 :: name0


    ! select backbone atoms
    
    do i = 1,nato
       res0 = r_label(i)%resseq
       name0 = r_label(i)%name
       if(res0 == res-1) then 
          if(trim(name0) == 'C') cm = i
       elseif(res0 == res) then 
          if(trim(name0) == 'N') n = i
          if(trim(name0) == 'CA') ca = i
          if(trim(name0) == 'C') c = i
       elseif(res0 == res+1) then 
          if(trim(name0) == 'N') np = i
       end if
    end do

    ! compute ramachandran angles
    ! first residue and last residue will have phi = 0 and psi = 0, respectively
    
    phi = 0.0
    psi = 0.0
    
    if(res /= 1) call atom_dihedral(nato,r,cm,n,ca,c,phi)
    if(res /= r_label(nato)%resseq) call atom_dihedral(nato,r,n,ca,c,np,psi)
    
  end subroutine ramachandran

  subroutine n_hbonds(nato,r,r_label,d0_pm,xnhb)
 
    !  returns the (real) number of H-bonds between BACKBONE atoms
    !  it is a func of distance only: see Bussi et al. JACS 2006
    !  input parameter: d0 in picometers


    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    type(label), intent(in) :: r_label(nato)
    integer(ki), intent(in) :: d0_pm
    ! local variables
    real(kr) :: xnhb
    real(kr) :: d,d0,d6,d12,f
    character*4 :: name1, name2
    integer(ki) :: i, j

    ! d0 is input in picometers
    d0 = d0_pm/100.

    
    xnhb = 0.
    ! select (O, H) pairs in the backbone

    do i = 1,nato
       name1 = r_label(i)%name
       if(name1 == 'O   ') then 
          do j = 1,nato
             name2 = r_label(j)%name
             if(name2 == 'H   ') then 
                call atom_distance(nato,r,i,j,d)
                d6 = ( d/d0 )**6
                d12 = ( d/d0 )**12
                f = (1-d6) / (1-d12)
                xnhb=xnhb+f
             end if
          end do
       end if
    end do

  end subroutine n_hbonds

  subroutine inertia_tensor(x,t)
    ! compute inertia tensor t for a set of n points with coord x
    implicit none
    real(kr), intent(in) :: x(:,:)
    real(kr), intent(out) :: t(3,3)
    ! local 
    integer(ki) :: i,j
    integer(ki) :: err
    integer(ki) :: n
    real(kr), allocatable :: d2(:),x1(:,:)

    n = size(x,2)
    allocate(d2(n),x1(3,n),stat=err)
    t = 0.0
    x1 = x
    ! put origin in c.o.m.
    do i = 1,3
       x1(i,:) = x1(i,:) - sum(x(i,:))/real(n)
    end do
    ! compute distance from c.o.m. for each point
    do i = 1,n
       d2(i) = sum(x1(:,i)**2)
    end do
    ! compute inertia tensor
    do i = 1,3
       t(i,i) = sum(d2 - x1(i,:)**2)
       do j = i+1,3
          t(i,j) = - sum(x1(i,:)*x1(j,:))
       end do
    end do
    deallocate(d2,x1)
    
  end subroutine inertia_tensor

  subroutine principal_axis(t,pa,pv)
    ! returns principal axis pa and values pv given an inertia tensor t
    implicit none
    real(kr), intent(in) :: t(3,3)
    real(kr), intent(out) :: pa(3,3),pv(3)
    ! local
    integer :: err
    integer(ki), parameter:: LWORK=1000
    real*8 :: WORK(LWORK)

    ! find eigenvectors and eigenvalues of inertia tensor

    pa = t

    call DSYEV( 'V', 'U', 3, pa, 3, pv, WORK, LWORK, err )

    if(err /= 0) stop 'error in principal axis'
    
  end subroutine principal_axis

  subroutine line_fit(r,at1,at2,v1,v2,v3)
    ! returns the line that best fits the positions of atoms subset at1 - at2
    implicit none
    real(kr), intent(in)    :: r(:,:) 
    integer(ki), intent(in) :: at1,at2
    real(kr), intent(out)   :: v1,v2,v3
    ! local variables
    integer(ki) :: err
    integer(ki) :: n
    real(kr), allocatable :: x(:,:)
    real(kr) :: t(3,3),pa(3,3),pv(3)

    n = at2-at1+1
    
    ! build coordinates matrix x
    
    allocate(x(3,n),stat=err)
    
    x = r(:,at1:at2)
    
    ! compute the inertia tensor
    
    call inertia_tensor(x,t)

    ! compute the principal axis of inertia
    
    call principal_axis(t,pa,pv)

    ! find the minimal inertia axis

    v1 = pa(1,1)
    v2 = pa(2,1)
    v3 = pa(3,1)

    deallocate(x,stat=err)

  end subroutine line_fit

  subroutine fit_normal(r,at1,at2,v1,v2,v3)
    ! returns the normal of the plane that best fits the positions of atoms in subset at1 - at2
    implicit none
    real(kr), intent(in)    :: r(:,:) 
    integer(ki), intent(in) :: at1,at2
    real(kr), intent(out)   :: v1,v2,v3
    ! local variables
    integer(ki)           :: err
    integer(ki)           :: n
    real(kr), allocatable :: x(:,:)
    real(kr)              :: t(3,3),pa(3,3),pv(3)

    
    n = at2-at1+1
    
    ! build coordinates matrix x
    
    allocate(x(3,n),stat=err)
    
    x = r(:,at1:at2)

    ! compute the inertia tensor
    
    call inertia_tensor(x,t)

    ! compute the principal axis of inertia
    
    call principal_axis(t,pa,pv)

    ! find the max inertia axis

    v1 = pa(1,3)
    v2 = pa(2,3)
    v3 = pa(3,3)

    deallocate(x,stat=err)

  end subroutine fit_normal

  subroutine centroid(r,at1,at2,v1,v2,v3)
    ! returns the centroid of atoms in subset at1 - at2
    implicit none
    real(kr), intent(in)    :: r(:,:) 
    integer(ki), intent(in) :: at1,at2
    real(kr), intent(out)   :: v1,v2,v3
    ! local variables
    integer(ki) :: i
    integer(ki) :: err
    integer(ki) :: n
    real(kr) :: c(3)
    real(kr), allocatable :: x(:,:)

    n = at2-at1+1

    ! build coordinates matrix x
    
    allocate(x(3,n),stat=err)
    
    x = r(:,at1:at2)
    
    do i = 1,3
       c(i) = sum(x(i,:))/real(n)
    end do

    v1 = c(1)
    v2 = c(2)
    v3 = c(3)

    deallocate(x,stat=err)

  end subroutine centroid

  subroutine centroid_rp(r,ats,v1,v2,v3)
    ! returns the centroid of atoms in subset at1 - at2
    implicit none
    real(kr), intent(in)    :: r(:,:) 
    integer(ki), intent(in) :: ats(6)
    real(kr), intent(out)   :: v1,v2,v3
    ! local variables
    integer(ki) :: i
    integer(ki) :: err
    integer(ki) :: n
    real(kr) :: c(3)
    real(kr) :: x(3,6)
    
    n=6

    ! build coordinates matrix x
    do i = 1,n
       x(:,i) = r(:,ats(i))
    end do
    
    do i = 1,3
       c(i) = sum(x(i,:))/real(n)
    end do

    v1 = c(1)
    v2 = c(2)
    v3 = c(3)

  end subroutine centroid_rp

  subroutine fit_normal_rp (r,ats,v1,v2,v3)
    ! ats dovrebbe essere un vettore che contiene
    ! il numero "di serie" degli atomi di cui calcolare
    ! il piano medio.
    ! returns the normal of the plane that best fits the positions of atoms in subset at1 - at2
    implicit none
    real(kr), intent(in)    :: r(:,:) 
    integer(ki), intent(in) :: ats(6)
    real(kr), intent(out)   :: v1,v2,v3
    ! local variables
    integer(ki) :: err
    integer(ki) :: n
    real(kr) :: x(3,6)
    real(kr) :: t(3,3),pa(3,3),pv(3)
    integer(ki) :: i
    
    do i = 1,6
       x(:,i) = r(:,ats(i))
!!$       write(*,*) x(:,i)
   end do
       

    ! compute the inertia tensor
    
    call inertia_tensor(x,t)

    ! compute the principal axis of inertia
    
    call principal_axis(t,pa,pv)

    ! find the max inertia axis

    v1 = pa(1,3)
    v2 = pa(2,3)
    v3 = pa(3,3)

  end subroutine fit_normal_rp


  subroutine mod_vect(V,m)
    implicit none
    real(kr),intent(in)  :: V(3)
    real(kr),intent(out) :: m
    ! local variables
    integer(ki) :: i
    
    m = 0.d0

    do i=1,3
       m = m + V(i)**2
    end do

    m = sqrt(m)

  end subroutine mod_vect

  subroutine scal_product(V1,V2,r)
    implicit none
    real(kr),intent(in)  :: V1(3),V2(3)
    real(kr),intent(out) :: r
    ! local variables
    integer(ki) :: i
    
    r = 0.d0

    do i = 1,3
       r = r + V1(i) * V2(i)
    end do

  end subroutine scal_product

  subroutine puckering(nato,r,r_label,res,q,theta,phi)
    ! returns the puckering parameters (q,theta,phi) of GLUCOSE residue r
    ! atoms are labeled C1-C5, O5
    ! Ref: D. Cremer and J. A. Pople, JACS 97,1354 (1975)
    implicit none
    integer(ki), intent(in) :: nato
    real(kr), intent(in)    :: r(3,nato) 
    type(label), intent(in) :: r_label(nato)
    integer(ki), intent(in) :: res
    real(kr),intent(out) :: q,theta,phi
    ! local variables 
    integer(ki) :: i,k
    integer(ki) :: np
    integer(ki) :: ats(6)
    real(kr) :: x(3,6),c1,c2,c3,n1,n2,n3,c(3),xb(3,6)
    real(kr) :: R1(3),R2(3),Rn(3),Rnm,z(6),qx,qy,qz
    real(kr) ::  PI
    real(kr) :: s
    
    PI = 4*DATAN(1.d0)
    
    np = 0
    do i = 1,nato 
       if(r_label(i)%resseq == res) then 
          select case(trim(adjustl(r_label(i)%name)))
          case('O5')
             x(:,1) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case('C1')
             x(:,2) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case('C2')
             x(:,3) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case('C3')
             x(:,4) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case('C4')
             x(:,5) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case('C5')
             x(:,6) = r(:,i)
             np = np + 1
             ats(np) = r_label(i)%serial
          case default
             ! do nothing
          end select
       end if
    end do
    
    if(np /= 6) then 
       write(0,*) "error: cannot find sugar ring atoms"
       write(0,*) "the program will stop"
       stop
    end if
    

    ! Sto usando il metodo "strano"
    ! ats -> Atomi dei quali calcolare il piano medio

    ! compute the centroid of the selected atoms
    call centroid_rp(r,ats,c1,c2,c3)
    
    c(1) = c1
    c(2) = c2
    c(3) = c3


    ! move the origin on the centroid
    
    do i = 1,6
       xb(:,i) = x(:,i) - c
    end do

    !normal of the plane that best fits the positions of atoms in ats
    R1=0.d0
    R2=0.d0

    do i = 1,6
       R1(:) = R1(:) + xb(:,i)*sin(2.d0*PI*(real(i-1)/real(6)))
       R2(:) = R2(:) + xb(:,i)*cos(2.d0*PI*(real(i-1)/real(6)))
    end do
    
    call cross_product(R1,R2,Rn)
    call mod_vect(Rn,Rnm)
    
    Rn(:) = Rn(:) / Rnm

    !    write(*,*) 'strage way to compute normal', rn
    !    call fit_normal_rp(r,ats,rn(1),rn(2),rn(3))
    !   write(*,*) 'reasonable way to compute normal', rn(1),rn(2),rn(3)
    !
    !   stop
    
    ! compute the dispacement between the plane and the atoms
    do i =1,6
       call scal_product(xb(:,i),Rn,z(i))
    end do

    ! Calcolo Q
    q = sqrt(sum(z**2))

    ! Calcolo q2cosphi2  q2sinphi2 q3
    qx = 0.d0
    qy = 0.d0
    qz = 0.d0
    
    do i = 1,6
       s = 1.0
       if(mod(i,2) == 0)  s = -1.0 
       qx = qx + z(i)*DCOS(4.d0*PI*(real(i-1))/6.d0)
       qy = qy + z(i)*DSIN(4.d0*PI*(real(i-1))/6.d0)
       qz = qz + z(i)*s
    end do

    qx = sqrt(2.d0/6.d0)*qx
    qy = -sqrt(2.d0/6.d0)*qy
    qz = qz / sqrt(6.d0) 

    theta = acos(qz/q) 

    phi = atan2(qy/(q*sin(theta)),qx/(q*sin(theta))) * 180.0 / pi
    
    theta = theta * 180.0 / pi
    if(phi < 0.0)then
       phi = phi + 360.0
    end if

  end subroutine puckering
  
end module cv
