#!/bin/bash
# example MBAR calculation from a REM run:
#   Ramachandran angles of alanine dipeptide
#

#! /bin/bash

#
# $0 DIR DCD_FILE DCD_REF 
#    
#    prepare MBAR input files from a REM run

# ----------------------------------------------------------------------
# help: copy first comments block in this file
# ----------------------------------------------------------------------
function help() {
name=`basename "$0"`;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------
ORAC_DIR=../..

# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "n:12-" Option
do
  case $Option in
#      1     )    test=1;;
#      2     )    test=2;;
      *     )    help;;   # help: copy first comments block
  esac
done


shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
test=$1
if [ $# -lt 1 ] ; then test=1;  fi

# ----------------------------------------------------------------------
# final parameters setting
# ----------------------------------------------------------------------
if [[ $test == 1 ]] ; then 
  echo "Will run test 1"; 
  REM_DIR=./sample-data
  PDB_REF_FILE=${REM_DIR}/diala-noh.pdb
  SCALING_FACTORS=${REM_DIR}/diala.REM_set
  ENERGIES=${REM_DIR}/diala.rem_ene
  PDB=${REM_DIR}/diala.r0xx.pdb
  PDB_R01=${REM_DIR}/diala.r001.pdb
  BIN_WIDTH=5.
fi

if [[ $test == 2 ]] ; then 
  echo "Will run test 2"; 
  REM_DIR=${ORAC_DIR}/tests/REM_tests
  DCD_FILE=1.dcd
  PDB_REF_FILE=../../pdb/DIALA.pdb
  SCALING_FACTORS=scaling_factors.dat
  ENERGIES=mbar.ene
  PDB=mbar.pdb
  PDB_R01=mbar.r01.pdb
  BIN_WIDTH=20.
fi

TEMPERATURE=300.


# ----------------------------------------------------------------------
# engine
# ----------------------------------------------------------------------


# make sure required tools have been compiled
if [[ $test == 2 ]] ; then 
    echo "compiling dcd2pdb ..."
    make -C ${ORAC_DIR}/tools/dcd2pdb     >/dev/null 2>&1
fi
echo "compiling analysis ..."
make -C ${ORAC_DIR}/tools/analysis    >/dev/null 2>&1
echo "compiling hist2d ..."
make -C ${ORAC_DIR}/tools/hist2d/src  >/dev/null 2>&1


# (0) first, run test 1 of REM 
if [[ $test == 2 ]] ; then 
    cd ${REM_DIR};
    make OUT_TEST_1
    cd -
# 
# (1) Prepare MBAR input files and .pdb files from REM run
#      
    echo "MBAR> Preparing MBAR input files from REM ..."
    if [ ! -r ${REM_DIR}/PAR0000/${DCD_FILE} ] ; then 
	echo "file ${REM_DIR}/PAR0000/${DCD_FILE} not found, or not readable. Exiting."
	exit
    fi
    set -o xtrace
    ./rem2mbar.sh ${REM_DIR} ${DCD_FILE} ${PDB_REF_FILE}
    set +o xtrace
fi


#
# (2) calculate mbar weights 
#
echo "MBAR> Starting MBAR ..."
set -o xtrace
./src/mbar ${SCALING_FACTORS} ${ENERGIES} ${TEMPERATURE}
set +o xtrace
echo "MBAR> ... MBAR done" 

#
# (3) extract one conformational variable from configurations
# 

# build the input for Ramachandran  calculation using analysis  
# 
cp $PDB mbar.r0xx.pdb 

# launch the analysis program 
../analysis/src/analysis < ana.r0xx.in   >/dev/null 2>&1
echo "MBAR> analysis for ramachandran done." 

#
# (4) match MBAR weights to conformational variable
#

# now paste ramachandran.01.out with mbar weights 
paste ramachandran.01.out mbar.weights | awk '{print $1,$2,$3}' > ramachandran.01.mbar
echo "MBAR> ramachandran angles and MBAR weights pasted and saved to 'ramachandran.01.mbar'" 

# (5) finally, make a MBAR-weighted histogram of variable
echo "MBAR> making histogram..."
set -o xtrace
${ORAC_DIR}/tools/hist2d/src/hist2d ramachandran.01.mbar mbar.plt ${BIN_WIDTH}
set +o xtrace
echo "MBAR> ... histogram created."


# (6) For comparison, make an unweighted histog of replica 1 only 

# build the input for Ramachandran  calculation using analysis  
# 
cp $PDB_R01 mbar.r001.pdb 

# launch the analysis program 
../analysis/src/analysis < ana.r001.in   >/dev/null 2>&1
echo "MBAR> analysis for ramachandran done." 

echo "MBAR> making second histogram..."
set -o xtrace
${ORAC_DIR}/tools/hist2d/src/hist2d ramachandran.01.out mbar.r01.plt ${BIN_WIDTH}
set +o xtrace
echo "MBAR> ... second histogram created."

echo "----------------------------------------------"
echo "MBAR Test completed."
echo "You may now want to run \`gnuplot fes-plot.gp' to plot a FES map"

