! Compute reweighting factors from ORAC multiple ensemble simulations
! using the MBAR method of Shirts and Chodera (J. Chem. Phys. y2008 v129 n124105).
!
! Syntax: mbar [scaling_factors_file [energies_file [Temperature]]]

! The scaling_factors_file contains:
!
! -  two comment lines
! -  then NREP lines, one for each ensemble, with 
!
!     <ensemble_index> <scale_fact_1> <scale_fact_2> <scale_fact_3> 
!
! Each row of the energies_file corresponds to one configuration sampled during the
! simulation, without any special order, and contains four fields:
!
!    <ensemble_index> <unscaled_potential_1> <unsc_pot_2> <unsc_pot_3> 
! 
!   potential energies are in kJ/mol

! Output file `mbar.weights' has one line per configuration:
!    
!    <weight> <ensemble_index>

! Dimensionless free energies (simple Bennett and MBAR) are printed to standard output


MODULE PRECISION_MODULE
! Real kinds 
  INTEGER, PARAMETER :: KR4 = SELECTED_REAL_KIND(6,37)       ! single precision real 
  INTEGER, PARAMETER :: KR8 = SELECTED_REAL_KIND(15,307)     ! double precision real 
  INTEGER, PARAMETER :: KR16 = SELECTED_REAL_KIND(30,1000)   ! quadruple precision real 
  integer, parameter :: krlong = selected_real_kind(18,400)  ! 18 digits precision, range of 10^-400 to 10^400
! Integer kinds 
  INTEGER, PARAMETER :: KI4 = SELECTED_INT_KIND(9)           ! single precision integer 
  INTEGER, PARAMETER :: KI8 = SELECTED_INT_KIND(18)          ! double precision integer 
! Complex kinds 
  INTEGER, PARAMETER :: KC4 = KR4                            ! single precision complex 
  INTEGER, PARAMETER :: KC8 = KR8                            ! double precision complex 
END MODULE PRECISION_MODULE

PROGRAM MBAR

  USE PRECISION_MODULE

  IMPLICIT NONE

! Tolerance in the iterative procedure
  REAL(8), PARAMETER :: DF_THRE = 1.E-7
  INTEGER, PARAMETER :: NCHAN = 500

  INTEGER :: IREP, NREP, P, NP, RP, ITER, N_UP, N_DOWN, I, CHAN, J, PPP, PTMP, IERROR, IT, NIR
  REAL(8), ALLOCATABLE, SAVE :: E(:,:), REM_FACT(:,:), B(:,:), F(:), F_OLD(:)
  REAL(8), ALLOCATABLE, SAVE :: D(:), EXPO(:), D1(:), W1(:), W_DOWN(:), W_UP(:)
  REAL(krlong), ALLOCATABLE, SAVE ::  W0(:)
  INTEGER, ALLOCATABLE, SAVE :: NCONF (:),  REP(:), N(:)
  REAL(8) :: IE1, IE2, IE3
  REAL(8) :: T0, B0, GASCON, SUMMA, DF, DF_MAX, DFV_MAX, ZERO, NU, ND, FF, NRM, DX
  REAL(8) :: H2(-NCHAN:NCHAN), F2(-NCHAN:NCHAN), H1(-NCHAN:NCHAN), F1(-NCHAN:NCHAN)
  REAL(8) :: H2MAX, H1MAX, XX, C, CTMP, DA, C_A_SMALL
  CHARACTER(80) :: FILEOUT
  integer(ki4):: err 
  integer(ki4) :: nargs
  character(len=128) :: arg
  integer(ki4),parameter :: upar=11
  integer(ki4),parameter :: udata=12
  character(len=128) :: fscal 
  character(len=128) :: fene
  logical :: file_exists

  !-------------- read arguments 

  ! default values
  fscal = "scaling_factors.dat"
  fene = "mbar.ene"
  T0 = 298.

  ! find the number of arguments passed on the command line
  nargs = iargc()

  ! optional arg #1 is name of scaling factors file
  if(nargs >= 1)      call getarg(1,fscal)

  ! optional arg #2 is name of energies file
  if(nargs >= 2)      call getarg(2,fene)

  ! optional arg #3 is Temperature (in K)
  if(nargs >= 3)  then
     call getarg(3,arg)
     read(arg,*) T0
  end if
 
! print out arguments
  write(0,*) "Temperature         = ", T0
  write(0,*) "scaling factors file: ", trim(fscal)
  write(0,*) "energies file       : ", trim(fene)

  !-------------------------------------------
  
! input files
  inquire(file=trim(fene), exist=file_exists)
  if (file_exists) then
     OPEN( UNIT=21, file=trim(fene), FORM='FORMATTED', status='OLD')
  else
     write(0,*)  "ERROR: input file '", trim(fene), "' not found. Stopping."
     stop
  end if

  inquire(file=trim(fscal), exist=file_exists)
  if (file_exists) then
     OPEN( UNIT=22, file=trim(fscal), FORM='FORMATTED', status='OLD')
  else
     write(0,*)  "ERROR: input file '", trim(fscal), "' not found. Stopping."
     stop
  end if




! output file (weight factors for the configurations)
  FILEOUT = 'mbar.weights'
  OPEN( UNIT=12, FILE=FILEOUT, FORM='FORMATTED' )



! Gas constant (J mol^-1 K^-1)
  GASCON = 8.314472

! Calculation of the total number of configurations for all ensembles
  NP = 0
10 READ(21,*,END=11) IT
  NP = NP + 1
  GOTO 10
11 REWIND(UNIT=21)


! get number of ensembles
  NREP = 0
  READ(22,*)
  READ(22,*)
54 READ(22,*,END=55) PTMP
  NREP = NREP + 1
  GOTO 54
55 REWIND(UNIT=22)

! allocate variables with NREP lenght
  ALLOCATE(N(NREP),STAT=IERROR)
  ALLOCATE(B(NREP,3),STAT=IERROR)
  ALLOCATE(F(NREP),STAT=IERROR)
  ALLOCATE(F_OLD(NREP),STAT=IERROR)
  ALLOCATE(REM_FACT(NREP,3),STAT=IERROR)
  ALLOCATE(NCONF(NREP),STAT=IERROR)

! read the scaling_factors from file scaling_factors.dat anfd writes them out  
  READ(22,*)
  READ(22,*)
  DO P = 1, NREP
     READ(22,*) PTMP, REM_FACT(P,1), REM_FACT(P,2), REM_FACT(P,3)
  ENDDO
  WRITE(*,*) NREP, " scaling factors read from `scaling_factors.dat'"

! before allocating check the configuration file is correct and count configurations per state

  DO P=1,NP
     read(21,*) irep  
     if(irep.ge.1.and.irep.le.NREP) THEN
        NCONF(irep)=NCONF(irep)+1
     ELSE 
        write(6,*) "FATAL ERROR: configuration", P, " has replica index beyond bounds! Replica index is ",irep  
     END IF
  END DO
  REWIND(21)


! write out confs found per state 
  DO  P=1,NREP 
     if(nconf(p).eq.0)  THEN 
        write(6,*) "FATAL ERROR: Replica", P, " not present in mbar.in file"  
!     ELSE
!        write(6,*) "CONFIGURATION OF STATE",p," ARE:", nconf(p)
     END IF
  END DO

! Now can safely allocate variables 

  ALLOCATE(REP(NP),STAT=IERROR)
  ALLOCATE(E(NP,3),STAT=IERROR)
  ALLOCATE(D(NP),STAT=IERROR)
  ALLOCATE(D1(NP),STAT=IERROR)
  ALLOCATE(W0(NP),STAT=IERROR)
  ALLOCATE(W1(NP),STAT=IERROR)
  ALLOCATE(W_UP(NP),STAT=IERROR)
  ALLOCATE(W_DOWN(NP),STAT=IERROR)
  ALLOCATE(EXPO(NP),STAT=IERROR)

  DO P = 1, NP
! IREP = ensemble index
! IE1, IE2, IE3 = Unscaled potential energies (in kJ/mol)
     READ(21,*,END=100) IREP, IE1, IE2, IE3
     REP(P) = IREP
! E(P) = Potential Energy (kJ/mol) of configuration P
     E(P,1) = IE1
     E(P,2) = IE2
     E(P,3) = IE3
  ENDDO

100 CONTINUE


! calculate inverse temperatures
  B0 = 1. / ( GASCON * 0.001 * T0 )
  DO P = 1, NREP
     B(P,1) = B0 * REM_FACT(P,1)
     B(P,2) = B0 * REM_FACT(P,2)
     B(P,3) = B0 * REM_FACT(P,3)
  ENDDO



! N(RP) = number of configurations referred to ensemble RP
  N(:) = 0
  DO P = 1, NP
     RP = REP(P)
     N(RP) = N(RP) + 1
  ENDDO



  WRITE(*,*)
  WRITE(*,*) 'Number of configurations', NP
  DO P = 1, NREP
     WRITE(*,*) 'Number of configurations in the ensemble ', P, ' : ', N(P)
  ENDDO



! Compute F_N - F_1 with  Bennett
  WRITE(*,*)
  WRITE(*,*) 'Calculation of F by Bennett method'
  WRITE(*,'(3(A,G20.12))') 'Ens. ', 1, ' f (dimensionless) = ', F(1), ' df (dimensionless) = ', 0.

  FF = 0.D0

  DO IREP = 2, NREP
     N_UP = 0
     N_DOWN = 0
     DO P = 1, NP
        RP = REP(P)
        IF ( RP == IREP - 1 ) THEN
           N_UP = N_UP + 1
           W_UP(N_UP) = (B(IREP,1) - B(IREP-1,1))*E(P,1)+(B(IREP,2) - B(IREP-1,2))*E(P,2)+(B(IREP,3)-B(IREP-1,3))* E(P,3)
        ELSEIF ( RP == IREP ) THEN
           N_DOWN = N_DOWN + 1
           W_DOWN(N_DOWN) = (B(IREP-1,1) - B(IREP,1)) * E(P,1) + (B(IREP-1,2)-B(IREP,2))*E(P,2)+(B(IREP-1,3)-B(IREP,3))*E(P,3)
        ENDIF
     ENDDO

     IF ( N_UP > 0 .AND. N_DOWN > 0 ) THEN
        CALL BNT( N_UP, N_DOWN, W_UP, W_DOWN, DA )
        ND = N_DOWN
        NU = N_UP
        DA = DA - LOG( ND / NU )
        FF = FF + DA
     ELSE
        DA = 0.
     ENDIF

     WRITE(*,'(3(A,G20.12))') 'Ens. ', IREP, ' f (dimensionless) = ', FF, ' df (dimensionless) = ', DA
! this is used to initialize MBAR computation
     F(IREP) = FF
  ENDDO





! Compute F_N - F_1 with MBAR

  WRITE(*,*)
  WRITE(*,*) 'Calculation of F by MBAR method'
  WRITE(*,*) 'Tolerance between subsequent iterations', DF_THRE

! initialize partition functions
  F(1) = 0.
  F_OLD(:) = F(:)
  ITER = 0
  DF_MAX = 0.


300 ITER = ITER + 1
  WRITE(*,'(A,1X,I5,1X,10G20.12)') 'Iteration ', ITER, DF_MAX, DFV_MAX, F(NREP)


! D(P) = logarithm of denominator in Eq. 11 Ref. JCP (see top of this file)
  DO P = 1, NP
! compute C taking care of overflow (see Appendix C1a in Ref. JCP)
     NIR = 0
     DO IREP = 1, NREP
        IF ( NIR == 0 ) THEN
           IF ( N(IREP) > 0 ) THEN
              C = F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) )
              NIR = 1
           ENDIF
        ELSE
           IF ( N(IREP) > 0 ) THEN
              CTMP = F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) )
              IF ( CTMP > C ) C = CTMP
           ENDIF
        ENDIF
     ENDDO

     DO IREP = 1, NREP
        IF ( N(IREP) > 0 ) THEN
           EXPO(IREP) = EXP( F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) ) - C )
        ELSE
           EXPO(IREP) = 0.
        ENDIF
     ENDDO

! summation in increasing order to avoid underflow (see app. C1a in Ref. JCP)
     IF ( NREP > 1 ) CALL SORT_REAL8 ( NREP, EXPO )

     SUMMA = 0.D0
     DO IREP = 1, NREP
        SUMMA = SUMMA + EXPO(IREP)
     ENDDO
     D(P) = C + LOG(SUMMA)

  ENDDO

! F(IREP) = adimensional free energy (Eq. 11 Ref. JCP)
  DO IREP = 1, NREP
! compute C taking care of overflow
     C = - ( B(IREP,1) * E(1,1) + B(IREP,2) * E(1,2) + B(IREP,3) * E(1,3) ) - D(1)
     DO P = 2, NP
        CTMP = - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) - D(P)
        IF ( CTMP > C ) C = CTMP
     ENDDO

     DO P = 1, NP
        EXPO(P) = EXP( - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) - D(P) - C )
     ENDDO

     IF ( NP > 1 ) CALL SORT_REAL8 ( NP, EXPO )

     SUMMA = 0.D0
     DO P = 1, NP 
        SUMMA = SUMMA + EXPO(P)
     ENDDO
     SUMMA = C + LOG(SUMMA)

! F(1) is constrained to zero at the end of each iteration
     IF ( IREP == 1 ) ZERO = - SUMMA
     F(IREP) = - SUMMA - ZERO
  ENDDO

  IF ( ITER == 1 ) THEN
     F_OLD(:) = F(:)
     GOTO 300
  ELSE

! convergence criterion
     DF_MAX = ABS( F(2) - F_OLD(2) ) / ABS( F(2) )
     DFV_MAX = F(2) - F_OLD(2)
     DO IREP = 3, NREP
        DF = ABS( F(IREP) - F_OLD(IREP) ) / ABS( F(IREP) )
        IF ( DF > DF_MAX ) DF_MAX = DF
        IF ( F(IREP) - F_OLD(IREP) > DFV_MAX ) DFV_MAX = F(IREP) - F_OLD(IREP)
     ENDDO
     IF ( DF_MAX < DF_THRE ) THEN
        GOTO 200
     ELSE
        F_OLD(:) = F(:)
        GOTO 300
     ENDIF

  ENDIF

200 CONTINUE

  DO IREP = 1, NREP
     IF ( IREP == 1 ) THEN
        WRITE(*,'(3(A,G20.12))') 'Ens. ', IREP, ' f (dimensionless) = ', F(IREP), ' df (dimensionless) = ', 0.
     ELSE
        WRITE(*,'(3(A,G20.12))') 'Ens. ', IREP, ' f (dimensionless) = ', F(IREP), ' df (dimensionless) = ', F(IREP) - F(IREP-1)
     ENDIF
  ENDDO


!=================================================================================
! CALCULATE THE WEIGHTS W0 AT THE TARGET TEMPERATURE FROM THE FREE ENTROPIES

! D(P) = logarithm of denominator in Eq. 14 Ref. JCP (see top of this file)
  DO P = 1, NP
! compute C taking care of overflow (see Appendix C1a in Ref. JCP)
     NIR = 0
     DO IREP = 1, NREP
        IF ( NIR == 0 ) THEN
           IF ( N(IREP) > 0 ) THEN
              C = F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) )
              NIR = 1
           ENDIF
        ELSE
           IF ( N(IREP) > 0 ) THEN
              CTMP = F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) )
              IF ( CTMP > C ) C = CTMP
           ENDIF
        ENDIF
     ENDDO

     DO IREP = 1, NREP
        IF ( N(IREP) > 0 ) THEN
           EXPO(IREP) = EXP( F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) + LOG( DBLE(N(IREP)) ) - C )
        ELSE
           EXPO(IREP) = 0.
        ENDIF
     ENDDO

! summation in increasing order to avoid underflow (see app. C1a in Ref. JCP)
     IF ( NREP > 1 ) CALL SORT_REAL8 ( NREP, EXPO )

     SUMMA = 0.D0
     DO IREP = 1, NREP
        SUMMA = SUMMA + EXPO(IREP)
     ENDDO
     D(P) = C + LOG(SUMMA)
  ENDDO


! Compute C_a of Eq. 14 Ref. JCP 
! compute C taking care of overflow 
  C = - B0 * ( E(1,1) + E(1,2) + E(1,3) ) - D(1)
  DO P = 2, NP
     CTMP = - B0 * ( E(P,1) + E(P,2) + E(P,3) ) - D(P)
     IF ( CTMP > C ) C = CTMP
  ENDDO

  DO P = 1, NP
     EXPO(P) = EXP( - B0 * ( E(P,1) + E(P,2) + E(P,3) ) - D(P) - C )
  ENDDO

  IF ( NP > 1 ) CALL SORT_REAL8 ( NP, EXPO )

  C_A_SMALL = 0.D0
  DO P = 1, NP 
     C_A_SMALL = C_A_SMALL + EXPO(P)
  ENDDO
  C_A_SMALL = EXP( C + LOG(C_A_SMALL) )

! compute weights W_na (second formula in Eq. 13 Ref. JCP)
  DO P = 1, NP
     W0(P) = EXP( - B0 * ( E(P,1) + E(P,2) + E(P,3) ) - D(P) )
  ENDDO
  W0(:) = W0(:) / C_A_SMALL
!  WRITE(*,*) 'C_A_SMALL', C_A_SMALL



! Print weights of configurations: W0()
  DO P = 1, NP
     WRITE(12,*) W0(P), REP(P)
  ENDDO 
  CLOSE(12)





! Alternative (direct) method to
! compute weights W_na (second formula in Eq. 13 Ref. JCP)
!  NRM = 0.D0
!  DO P = 1, NP
!     SUMMA = 0.D0
!     DO IREP = 1, NREP
!        SUMMA = SUMMA + EXP( F(IREP) - ( B(IREP,1) * E(P,1) + B(IREP,2) * E(P,2) + B(IREP,3) * E(P,3) ) ) * N(IREP)
!     ENDDO
!     D1(P) = SUMMA
!     W1(P) = EXP( - B0 * ( E(P,1) + E(P,2) + E(P,3) ) ) / D1(P)
!     NRM = NRM + W1(P)
!  ENDDO
!  W1(:) = W1(:) / NRM
!  WRITE(*,*) 'NRM', NRM


  STOP


END PROGRAM MBAR

  



!===================================   CALCOLO DF CON BENNETT  =========================================
! compute the free energy difference dimensionless
SUBROUTINE BNT ( NWF, NWB, WF, WB, DA )

  USE PRECISION_MODULE

  IMPLICIT NONE

  INTEGER, INTENT(IN) :: NWF,NWB
  REAL(KR8), INTENT(IN) :: WF(NWF), WB(NWB)
  REAL(KR8), INTENT(OUT) :: DA
  REAL(KR8), EXTERNAL :: RTSAFE

  DA = RTSAFE( NWF, NWB, WF, WB, -5000.0_KR8, 5000.0_KR8, 0.0001_KR8 )

END SUBROUTINE BNT
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
FUNCTION RTSAFE ( NWF, NWB, WF, WB, X1, X2, XACC )

  USE PRECISION_MODULE

  INTEGER, INTENT(IN) :: NWF, NWB
  REAL(KR8), INTENT(IN) :: WF(NWF), WB(NWB)
  INTEGER :: MAXIT
  REAL(KR8) :: RTSAFE, X1, X2, XACC
  PARAMETER ( MAXIT = 100 )
  INTEGER :: J
  REAL(KR8) :: DF, DX, DXOLD, F, FH, FL, TEMP, XH, XL
    
  CALL FUNCD( NWF, NWB, WF, WB, X1, FL, DF )
  CALL FUNCD( NWF, NWB, WF, WB, X2, FH, DF )

  IF ( ( FL > 0.0_KR8 .AND. FH > 0.0_KR8 ) .OR. ( FL < 0.0_KR8 .AND. FH < 0.0_KR8 ) ) WRITE(*,*) 'root must be bracketed in rtsafe'
  IF ( FL == 0.0_KR8 ) THEN
     RTSAFE = X1
     RETURN
  ELSEIF ( FH == 0.0_KR8 ) THEN
     RTSAFE = X2
     RETURN
  ELSEIF ( FL < 0.0_KR8 ) THEN
     XL = X1
     XH = X2
  ELSE
     XH = X1
     XL = X2
  ENDIF
  RTSAFE = 0.5_KR8 * ( X1 + X2 )
  DXOLD = ABS( X2 - X1 )
  DX = DXOLD
  CALL FUNCD( NWF, NWB, WF, WB, RTSAFE, F, DF )
  DO J = 1, MAXIT
     IF ( ( (RTSAFE-XH)*DF-F)*((RTSAFE-XL)*DF-F) >= 0.0_KR8 .OR. ABS(2.*F) > ABS(DXOLD*DF) ) THEN 
        DXOLD = DX
        DX = 0.5_KR8 * ( XH - XL )
        RTSAFE = XL + DX
        IF ( XL == RTSAFE ) RETURN
     ELSE
        DXOLD = DX
        DX = F / DF
        TEMP = RTSAFE
        RTSAFE = RTSAFE - DX
        IF ( TEMP == RTSAFE ) RETURN
     ENDIF
     IF ( ABS(DX) < XACC ) RETURN
     CALL FUNCD( NWF, NWB, WF, WB, RTSAFE, F, DF )
     IF ( F < 0.0_KR8 ) THEN
        XL = RTSAFE
     ELSE
        XH = RTSAFE
     ENDIF
  ENDDO
  WRITE(*,*) 'RTSAFE EXCEEDING MAXIMUM ITERATIONS'
  RETURN
END FUNCTION RTSAFE
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE FUNCD ( NWF, NWB, WF, WB, DA, LL, LLD )

  USE PRECISION_MODULE

  IMPLICIT NONE

  INTEGER, INTENT(IN) :: NWF,NWB
  REAL(KR8), INTENT(IN) :: WF(NWF), WB(NWB)
  REAL(KR8), INTENT(IN) :: DA
  REAL(KR8), INTENT(OUT) :: LL,LLD
  REAL(KR8), PARAMETER :: EPS = 0.00001_KR8
  REAL(KR8), EXTERNAL :: DLLIKE
    
  LL = DLLIKE( NWF, NWB, WF, WB, DA )
  LLD = ( DLLIKE( NWF, NWB, WF, WB, DA + EPS ) - DLLIKE( NWF, NWB, WF, WB, DA - EPS ) ) / ( 2.0_KR8 * EPS )
    
END SUBROUTINE FUNCD
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
FUNCTION DLLIKE ( NWF, NWB, WF, WB, DA )

  USE PRECISION_MODULE

  IMPLICIT NONE

  INTEGER, INTENT(IN) :: NWF,NWB
  REAL(KR8), INTENT(IN) :: WF(NWF), WB(NWB)
  REAL(KR8), INTENT(IN) :: DA
  REAL(KR8) :: DLLIKE

  DLLIKE = SUM( 1.0_KR8 / (1.0_KR8 + EXP(WF - DA)) ) - SUM( 1.0_KR8 / (1.0_KR8 + EXP(WB + DA)) )
    
END FUNCTION DLLIKE
!===================================   CALCOLO DF CON BENNETT (fine) ====================================

! Order RA in increasing order
SUBROUTINE SORT_REAL8(N,RA)

  IMPLICIT NONE

  INTEGER :: L, J, IR, I, N
  REAL(8) :: RA(N), RRA

  L=N/2+1                                                                   
  IR=N                                                                      
10 CONTINUE                                                                  
  IF(L.GT.1)THEN                                                          
     L=L-1                                                                 
     RRA=RA(L)                                                             
  ELSE                                                                    
     RRA=RA(IR)                                                            
     RA(IR)=RA(1)                                                          
     IR=IR-1                                                               
     IF(IR.EQ.1)THEN                                                       
        RA(1)=RRA                                                           
        RETURN                                                              
     ENDIF
  ENDIF
  I=L                                                                     
  J=L+L                                                                   
20 IF(J.LE.IR)THEN                                                         
     IF(J.LT.IR)THEN                                                       
        IF(RA(J).LT.RA(J+1))J=J+1                                           
     ENDIF
     IF(RRA.LT.RA(J))THEN                                                  
        RA(I)=RA(J)                                                         
        I=J                                                                 
        J=J+J                                                               
     ELSE                                                                  
        J=IR+1                                                              
     ENDIF
     GO TO 20                                                                
  ENDIF
  RA(I)=RRA                                                               
  GOTO 10                                                                  
END SUBROUTINE SORT_REAL8

! Order RA in increasing order
SUBROUTINE SORT_INTEGER(N,RA)

  IMPLICIT NONE

  INTEGER :: L, J, IR, I, N
  INTEGER :: RA(N), RRA

  L=N/2+1                                                                   
  IR=N                                                                      
10 CONTINUE                                                                  
  IF(L.GT.1)THEN                                                          
     L=L-1                                                                 
     RRA=RA(L)                                                             
  ELSE                                                                    
     RRA=RA(IR)                                                            
     RA(IR)=RA(1)                                                          
     IR=IR-1                                                               
     IF(IR.EQ.1)THEN                                                       
        RA(1)=RRA                                                           
        RETURN                                                              
     ENDIF
  ENDIF
  I=L                                                                     
  J=L+L                                                                   
20 IF(J.LE.IR)THEN                                                         
     IF(J.LT.IR)THEN                                                       
        IF(RA(J).LT.RA(J+1))J=J+1                                           
     ENDIF
     IF(RRA.LT.RA(J))THEN                                                  
        RA(I)=RA(J)                                                         
        I=J                                                                 
        J=J+J                                                               
     ELSE                                                                  
        J=IR+1                                                              
     ENDIF
     GO TO 20                                                                
  ENDIF
  RA(I)=RRA                                                               
  GOTO 10                                                                  
END SUBROUTINE SORT_INTEGER

