#!/bin/bash

# Syntax: run_serial_tests.bash [options] <ORAC> 
#  where <ORAC> is the full name of the program 
#   options: -z -> clear temp files and exit
#            -h -> display this help

# ORAC program: Do all Alchemical tests sequentially 
# script has been tested on Linux platforms only. 

# ----------------------------------------------------------------------
# Help: copy first comments block in this file
# ----------------------------------------------------------------------
function Help {
name=`basename "$0"`;
sed -n "s/\$0/$name/;2,/^\$/p" $0; exit
}


# ----------------------------------------------------------------------
# other functions
# ----------------------------------------------------------------------
function CleanUpFiles {
    rm -rf $* >& /dev/null
}

# ----------------------------------------------------------------------
# init and defaults   
# ----------------------------------------------------------------------
TEMP_FILES="\
  OUT_TEST \
  *.out \
  *.pdb \
  *.tpgprm \
  */*.rst \
  *.rst \
  *.rst* \
  tmp* \
  *.works* \
  *.wrk \
  wr* \
  PAR* \
  works* \
  SWITCH*/*.* \
  REST*/*.* \
   WR* \
  test.tt* \
  *~ \
"



# ----------------------------------------------------------------------
# command line options
# ----------------------------------------------------------------------
while getopts "hz" Option
do
  case $Option in
    h     )    Help;;    # help: copy first comments block
    z     )    CleanUpFiles $TEMP_FILES; exit ;;  # just cleanup
    *     )    Help;;
  esac
done

shift $(($OPTIND - 1))
#  Decrements the argument pointer so it points to next argument.
#  $1 now references the first non option item supplied on the command line
#+ if one exists.

# ----------------------------------------------------------------------
# command line arguments
# ----------------------------------------------------------------------
if [ $# -lt 1 ] ; then
    Help;
    exit
else 
    ORAC=$1
fi


# ----------------------------------------------------------------------
# initial checks and cleanup
# ----------------------------------------------------------------------

#
# make sure all needed programs are executable and on the $PATH
#
###BINARIES="mpiexec mpdallexit mpdboot $ORAC"
BINARIES="$ORAC"

for prog in $BINARIES; do
    if ! type $prog >/dev/null 2>&1; then
	echo "ERROR: \"$prog\" not found."
	echo "       This is needed by $0 to work. Check your"
	echo "       \$PATH variable or install program \"$prog\"."
	exit 2
    fi
done


#
# files cleanup
#
CleanUpFiles $TEMP_FILES;

# ----------------------------------------------------------------------
#   TESTS
# ----------------------------------------------------------------------

echo "Doing first test for Alchemical Transformation ....." 
echo " PLATFORM = " `uname -a`   > OUT_TEST
echo " ORAC executable = " $ORAC   >> OUT_TEST
echo " RELEASE = " `cat ../../RELEASE` >> OUT_TEST
  # First test 
  # Test on driven stretching, bending and torsion of decaalanine
echo "Testing 1.in: switching ON of N-Elte378 in liquid hexane..."
echo "Testing 1.in..." >> OUT_TEST
$ORAC < 1.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 

echo "Testing 1r.in: switching ON of N-Elte378 in liquid hexane (restarted).."
echo "Testing 1r.in..." >> OUT_TEST
$ORAC < 1r.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 
echo "Test 1 done; alchemical work file is in alch1.wrk"

echo "Testing 2.in: switching OFF of N-Elte378 in liquid hexane..."
echo "Testing 2.in..." >> OUT_TEST
$ORAC < 2.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 

echo "Testing 2r.in: switching OFF of N-Elte378 in liquid hexane (restarted)..."
echo "Testing 2r.in..." >> OUT_TEST
$ORAC < 2r.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 
echo "Test 2 done; alchemical work file is in alch2.wrk"

echo "Testing 3.in: switching ON and OFF of two N-Elte378 in liquid hexane..."
echo "Testing 3.in..." >> OUT_TEST
$ORAC < 3.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 

echo "Testing 3r.in: : switching ON and OFF of two N-Elte378 in liquid hexane (restarted)..."
echo "Testing 3r.in..." >> OUT_TEST
$ORAC < 3r.in >& tmp ;
grep "Total    = " tmp >> OUT_TEST; rm tmp; 
echo "Test 3 done; alchemical work file is in alch3.wrk"


# Now check if tests are OK (or almost OK) 
diff OUT_TEST OUT | grep -v PLATFORM >& tmp.diff;

echo "** TEST COMPLETED **"

if `test -s tmp.diff`; 
then 
    echo "-------------------------------------------------------------------"
    echo " Warning - your output differs from reference:" 
    echo " compare OUT_TEST to reference OUT (try \`make diff'), "
    echo " or you may want to run individual tests with e.g. \`make 2b.out'"
    echo ""
    echo " Please read the NOTE on tests in the README file"
    echo "-------------------------------------------------------------------"
    exit  
fi

exit
